use log::*;
use regex::Regex;
use serde::{
    de::{self, MapAccess, Visitor},
    ser::{SerializeMap, SerializeSeq, SerializeStruct},
    Deserialize, Deserializer, Serializer,
};
use std::{
    cell::{RefCell, RefMut},
    cmp::Ordering,
    collections::HashMap,
    convert::{Infallible, TryFrom, TryInto},
    fmt::{self, Debug},
    ops::{Deref, DerefMut},
    str::FromStr,
    sync::Arc,
};
pub(crate) mod dom;
pub use url::Url;
pub mod temporal;

#[cfg(test)]
mod test;

/// A helper type for representing a list of [property values][PropertyValue].
pub type NodeList = Vec<PropertyValue>;

/// A helper type for representing a map of named [property values][PropertyValue].
pub type Properties = ThreadSafeProperty<HashMap<String, NodeList>>;

/// A concrete reference of the supported Microformats class by this library.
#[derive(serde::Serialize, serde::Deserialize, Clone, Debug, PartialEq, Eq, PartialOrd, Ord)]
#[serde(rename_all = "kebab-case")]
pub enum KnownClass {
    /// Represents a general container for content.
    /// https://microformats.org/wiki/h-entry
    #[serde(alias = "h-entry")]
    Entry,

    /// Represents a rewference to content at a different location.
    /// https://microformats.org/wiki/h-cite
    #[serde(alias = "h-cite")]
    Cite,

    /// Represents a contact card or vCard.
    /// https://microformats.org/wiki/h-card
    #[serde(alias = "h-card")]
    Card,

    /// https://microformats.org/wiki/h-feed
    #[serde(alias = "h-feed")]
    Feed,

    /// https://microformats.org/wiki/h-event
    #[serde(alias = "h-event")]
    Event,

    /// https://microformats.org/wiki/h-product
    #[serde(alias = "h-product")]
    Product,

    /// https://microformats.org/wiki/h-adr
    #[serde(alias = "h-adr")]
    Adr,

    /// https://microformats.org/wiki/h-geo
    #[serde(alias = "h-geo")]
    Geo,

    /// https://microformats.org/wiki/h-resume
    #[serde(alias = "h-resume")]
    Resume,

    /// https://microformats.org/wiki/h-review
    #[serde(alias = "h-review")]
    Review,

    /// https://microformats.org/wiki/h-recipe
    #[serde(alias = "h-recipe")]
    Recipe,
}

impl FromStr for KnownClass {
    type Err = super::Error;

    /// Converts this concrete known class into a string.
    fn from_str(s: &str) -> Result<Self, Self::Err> {
        match s.to_ascii_lowercase().as_str() {
            "h-entry" | "entry" => Ok(Self::Entry),
            "h-cite" | "cite" => Ok(Self::Cite),
            "h-card" | "card" => Ok(Self::Card),
            "h-event" | "event" => Ok(Self::Event),
            "h-product" | "product" => Ok(Self::Product),
            "h-feed" | "feed" => Ok(Self::Feed),
            "h-geo" | "geo" => Ok(Self::Geo),
            "h-adr" | "adr" => Ok(Self::Adr),
            "h-recipe" | "recipe" => Ok(Self::Recipe),
            _ => Err(super::Error::NotKnownClass(s.to_string())),
        }
    }
}

impl ToString for KnownClass {
    /// Converts a known class into a string.
    fn to_string(&self) -> String {
        match self {
            KnownClass::Entry => "h-entry".to_owned(),
            KnownClass::Cite => "h-cite".to_owned(),
            KnownClass::Card => "h-card".to_owned(),
            KnownClass::Feed => "h-feed".to_owned(),
            KnownClass::Event => "h-event".to_owned(),
            KnownClass::Product => "h-product".to_owned(),
            KnownClass::Adr => "h-adr".to_owned(),
            KnownClass::Geo => "h-geo".to_owned(),
            KnownClass::Resume => "h-resume".to_owned(),
            KnownClass::Review => "h-review".to_owned(),
            KnownClass::Recipe => "h-recipe".to_owned(),
        }
    }
}

/// Represents a Microformat class.
/// https://microformats.org/wiki/Category:Draft_Specifications
#[derive(Debug, Clone, Eq)]
pub enum Class {
    /// Represents a known Microformat class (h-entry, h-card, etc).
    Known(KnownClass),

    /// Represents a class that's not spec-compliant (h-cookies, h-monster, etc).
    Unrecognized(String),

    /// Represents a custom vendor specific or experimental Microformat class (h-x-toy)
    // TODO: Correct this to expect two strings (vendor prefix and type)
    VendorSpecific(String),
}

impl PartialOrd for Class {
    fn partial_cmp(&self, other: &Self) -> std::option::Option<std::cmp::Ordering> {
        self.to_string().partial_cmp(&other.to_string())
    }
}

impl PartialEq for Class {
    fn eq(&self, other: &Self) -> bool {
        self.to_string().eq(&other.to_string())
    }
}

impl FromStr for Class {
    type Err = Infallible;

    /// Parses a string as a Microformat class.
    ///
    /// # Examples
    /// ```
    /// # use std::str::FromStr;
    /// # use microformats::types::{Class, KnownClass};
    ///
    /// assert_eq!(Class::from_str("entry"), Ok(Class::Known(KnownClass::Entry)));
    /// assert_eq!(Class::from_str("h-card"), Ok(Class::Known(KnownClass::Card)));
    /// assert_eq!(Class::from_str("h-x-plane"), Ok(Class::VendorSpecific("plane".to_string())));
    /// ```
    fn from_str(class_str: &str) -> Result<Self, Self::Err> {
        if class_str.starts_with("h-x-") {
            Ok(Self::VendorSpecific(
                class_str.get(4..).unwrap_or_default().to_string(),
            ))
        } else {
            match KnownClass::from_str(class_str)
                .or_else(|_| KnownClass::from_str(&class_str.replace("h-", "")))
            {
                Ok(known_class) => Ok(Self::Known(known_class)),
                Err(_) => Ok(Self::Unrecognized(
                    class_str.trim_start_matches("h-").to_string(),
                )),
            }
        }
    }
}

impl ToString for Class {
    /// Converts a Microformat class representation to its string form.
    fn to_string(&self) -> String {
        match self {
            Self::Known(class) => class.to_string(),
            Self::Unrecognized(class) => format!("h-{}", class),
            Self::VendorSpecific(class) => format!("h-x-{}", class),
        }
    }
}

impl serde::Serialize for Class {
    /// Serializes this class reference into a string.
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        serializer.serialize_str(self.to_string().as_str())
    }
}

impl Class {
    /// Checks if this class is a recognized one by the Microformats spec.
    pub fn is_recognized(&self) -> bool {
        !matches!(self, Self::Unrecognized(_))
    }
}

struct ClassVisitor;

impl<'de> Visitor<'de> for ClassVisitor {
    type Value = Class;

    fn expecting(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
        formatter.write_str("a string that follows Microformats class conventions")
    }

    fn visit_str<E>(self, class_str: &str) -> Result<Self::Value, E>
    where
        E: serde::de::Error,
    {
        Class::from_str(class_str).map_err(|e| E::custom(e.to_string()))
    }
}

impl<'de> serde::Deserialize<'de> for Class {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        deserializer.deserialize_string(ClassVisitor)
    }
}

fn short_circuit_url_deserialization<'de, D>(d: D) -> Result<Url, D::Error>
where
    D: serde::Deserializer<'de>,
{
    let string_form = String::deserialize(d)?;
    let url_form = Url::parse(&string_form).map_err(serde::de::Error::custom)?;

    if url_form.as_str() != string_form {
        // This is called in the event a string happens to match the parsing of a URL but doesn't
        // convert back into one.
        Err(serde::de::Error::custom(
            "This string doesn't represent a valid URL due looking like one.",
        ))
    } else {
        Ok(url_form)
    }
}

fn short_circuit_plain_text_deserialization<'de, D>(d: D) -> Result<String, D::Error>
where
    D: serde::Deserializer<'de>,
{
    let string_form = String::deserialize(d)?;
    trace!(
        "Checking if {:?} can be represented as a different value.",
        string_form
    );

    Url::from_str(&string_form)
        .map_err(serde::de::Error::custom)
        .map(|u| u.as_str().to_string())
        .and_then(|u| {
            if u == string_form && !u.contains(|c: char| c.is_whitespace()) && !u.contains('\n') {
                trace!("{:?} is actually a URL", u);
                Err(serde::de::Error::invalid_type(
                    de::Unexpected::Other("URL"),
                    &"plain 'ol string",
                ))
            } else {
                Ok(string_form.clone())
            }
        })
        .or_else(|r: D::Error| {
            if r.to_string().starts_with("invalid type: URL") {
                Err(r)
            } else {
                temporal::Value::from_str(&string_form)
                    .map_err(serde::de::Error::custom)
                    .map(|u| u.to_string())
                    .and_then(|u| {
                        if u == string_form {
                            trace!("{:?} is actually a temporal value", string_form);
                            Err(serde::de::Error::invalid_type(
                                de::Unexpected::Str("temporal data"),
                                &"plain 'ol string",
                            ))
                        } else {
                            Ok(string_form.clone())
                        }
                    })
            }
        })
        .or_else(|r: D::Error| {
            if r.to_string().starts_with("invalid type: URL")
                || r.to_string().contains("temporal data")
            {
                Err(r)
            } else {
                trace!("Looks like {:?} is just a string.", string_form);
                Ok(string_form)
            }
        })
}

/// Represents the multiple forms in which a property's value is represented.
#[derive(Debug, Clone, PartialEq, Eq, serde::Serialize, serde::Deserialize, PartialOrd, Ord)]
#[serde(untagged, rename_all = "kebab-case")]
pub enum PropertyValue {
    /// Represents plain-text, usually stored in properties like "p-name". It can be
    /// displayed directly to the user agent.
    /// <https://microformats.org/wiki/microformats2-parsing#parsing_a_p-_property>
    #[serde(deserialize_with = "short_circuit_plain_text_deserialization")]
    Plain(String),

    /// Represents a linked value.
    /// It's a companion of `Plain`; meant to make resolving non-absolute URLs easier.
    #[serde(deserialize_with = "short_circuit_url_deserialization")]
    Url(Url),

    /// Represents a compatible datetime parser defined by <https://microformats.org/wiki/value-class-pattern#Date_and_time_parsing>
    /// <https://microformats.org/wiki/microformats2-parsing#parsing_a_dt-_property>
    Temporal(temporal::Value),

    Fragment(Fragment),

    Image(Image),

    /// Represents a structured form of information presented by Microformats as an
    /// `Item`. This will usually require a bit more processing before showing it.
    /// <https://microformats.org/wiki/microformats2-parsing#parsing_a_u-_property>
    #[serde(with = "referenced_item")]
    Item(Item),
}

impl From<Item> for PropertyValue {
    fn from(item: Item) -> Self {
        PropertyValue::Item(item)
    }
}

/// Represents markup and the plain text representation accompanying it.
/// <https://microformats.org/wiki/microformats2-parsing#parsing_a_e-_property>
#[derive(
    Debug, Clone, PartialEq, Eq, serde::Serialize, Default, serde::Deserialize, PartialOrd, Ord,
)]
#[serde(rename_all = "kebab-case")]
pub struct Fragment {
    /// Provides the HTML representation of this fragment.
    #[serde(skip_serializing_if = "String::is_empty")]
    pub html: String,

    /// Provides the plain-text form of the HTML.
    #[serde(default, skip_serializing_if = "String::is_empty")]
    pub value: String,

    /// Provides the language that this fragment is represented in.
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub lang: Option<String>,
}

impl Fragment {
    /// Determines if this fragment has any information represented in it.
    ///
    /// The presence of HTML does not determine if this fragment is empty; a
    /// fragment can be defined with no HTML (meaning the HTML would implictly look the same).
    pub fn is_empty(&self) -> bool {
        self.value.is_empty()
    }
}

/// Represents the structured form of an image.
/// <https://microformats.org/wiki/microformats2-parsing#parse_an_img_element_for_src_and_alt>
#[derive(Debug, Clone, PartialEq, Eq, serde::Deserialize, serde::Serialize, PartialOrd, Ord)]
#[serde(rename_all = "kebab-case")]
pub struct Image {
    #[serde(alias = "value", alias = "src")]
    pub src: Url,

    #[serde(skip_serializing_if = "String::is_empty")]
    pub alt: String,
}

mod referenced_item {

    use super::*;

    type Value = Item;

    struct ItemVisitor;

    #[derive(serde::Deserialize, Debug)]
    #[serde(field_identifier, rename_all = "kebab-case")]
    enum ItemDeserializationFields {
        Children,
        Value,
        Id,
        Properties,
        r#Type,
    }

    impl<'de> Visitor<'de> for ItemVisitor {
        type Value = Value;
        fn expecting(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
            formatter.write_str("expecting null or an map representing an item")
        }

        fn visit_map<A>(self, mut item_map: A) -> Result<Self::Value, A::Error>
        where
            A: MapAccess<'de>,
        {
            let mut children: Vec<Item> = Default::default();
            let mut value: ThreadSafeProperty<Option<ValueKind>> = Default::default();
            let mut id: ThreadSafeProperty<Option<String>> = Default::default();
            let mut types = Vec::new();
            let properties = Properties::default();

            while let Some(property) = item_map.next_key()? {
                match property {
                    ItemDeserializationFields::Children => {
                        trace!("Adding in children");
                        let new_items = item_map.next_value::<Vec<Item>>()?.into_iter();

                        if children.is_empty() && new_items.len() > 0 {
                            children = new_items.collect();
                        } else {
                            children.extend(new_items);
                        }
                    }
                    ItemDeserializationFields::Value => {
                        trace!("Adding in value");
                        if value.0.borrow().is_none() {
                            value = ThreadSafeProperty::new(
                                item_map.next_value::<Option<ValueKind>>()?,
                            );
                        }
                    }
                    ItemDeserializationFields::Id => {
                        trace!("Adding in ID");
                        if id.0.borrow().is_none() {
                            id = ThreadSafeProperty::new(item_map.next_value::<Option<String>>()?);
                        }
                    }
                    ItemDeserializationFields::Type => {
                        trace!("Adding in types");
                        types.extend(item_map.next_value::<Vec<Class>>()?);
                    }
                    ItemDeserializationFields::Properties => {
                        trace!("Add in properties.");
                        properties
                            .0
                            .deref()
                            .borrow_mut()
                            .extend(item_map.next_value::<HashMap<String, _>>()?);
                    }
                }
            }

            let mut item = Item {
                parent: Arc::new(RefCell::new(ParentRelationship::Root)),
                r#type: types,
                properties: properties.into(),
                id,
                value,
                children: Arc::new(RefCell::new(children)),
                ..Default::default()
            };

            item.reassociate_parents();

            Ok(item)
        }
    }

    pub fn serialize<S>(item: &Value, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: Serializer,
    {
        serializer.serialize_some(&Some(item))
    }

    pub fn deserialize<'de, D>(deserializer: D) -> Result<Value, D::Error>
    where
        D: Deserializer<'de>,
    {
        trace!("Attempting to deserialize a referenced item.");
        deserializer.deserialize_struct(
            "Item",
            &["type", "properties", "id", "value", "children"],
            ItemVisitor,
        )
    }
}

impl PropertyValue {
    /// Determines if this node's internal value is empty
    pub fn is_empty(&self) -> bool {
        match self {
            Self::Temporal(_) | Self::Url(_) => false,
            Self::Image(_) => false,
            Self::Plain(s) => s.is_empty(),
            Self::Fragment(f) => f.is_empty(),
            Self::Item(i) => i.is_empty(),
        }
    }
}

impl From<Url> for PropertyValue {
    fn from(u: Url) -> Self {
        Self::Url(u)
    }
}

impl From<temporal::Stamp> for PropertyValue {
    fn from(t: temporal::Stamp) -> Self {
        Self::Temporal(temporal::Value::Timestamp(t))
    }
}

impl From<temporal::Duration> for PropertyValue {
    fn from(t: temporal::Duration) -> Self {
        Self::Temporal(temporal::Value::Duration(t))
    }
}

#[derive(Clone, PartialEq, Eq, PartialOrd, Ord, Debug)]
pub enum RelationshipKind {
    /// p-
    Plain(String),
    /// u-
    URL(String),
    /// dt-
    Datetime(String),
    /// e-
    HTML(String),
}

/// Represents the relationship an item has to its parent (if any).
#[derive(Clone, PartialEq, Eq, PartialOrd, Ord)]
pub enum ParentRelationship {
    /// Another item is the direct parent of this item (either via `Item::children` or as a direct
    /// item in a list of values in `Item::properties`).
    Child(Item),

    /// Another item is the direct parent of this item by way of a property type.
    Property(RelationshipKind, Item),

    /// A top-level item; the document itself is the parent.
    // FIXME: Should this have a reference to the actual document?
    Root,
}

impl Debug for ParentRelationship {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            Self::Root => f.write_str("Root"),
            Self::Child(item) => f.write_str(&format!(
                "ChildOfItem({:?})",
                item.id().or_else(|| Some(format!("{:?}", item.value())))
            )),
            Self::Property(kind, item) => f.write_str(&format!(
                "PropertyOf({:?}) as {:?}",
                item.id().or_else(|| Some(format!("{:?}", item.value()))),
                kind
            )),
        }
    }
}

impl ParentRelationship {
    /// Creates a new child element for this item parent.
    pub fn create_child_item(&mut self, document: &mut Document, types: &[Class]) -> Item {
        let base = self.to_owned();
        match self {
            Self::Root => document.create_child_item(types),
            Self::Child(ref mut parent_item) | Self::Property(_, ref mut parent_item) => {
                let new_item = Item::new(base, types.to_vec());
                parent_item.append_child(&new_item);
                new_item
            }
        }
    }

    /// Creates a new top-level item or resolves the current one.
    pub fn resolve_item(&self) -> Option<&Item> {
        match self {
            Self::Root => None,
            Self::Child(current_item) => Some(current_item),
            Self::Property(_, current_item) => Some(current_item),
        }
    }

    /// Creates a new top-level item or resolves the current one.
    pub(crate) fn resolve_item_mut(&mut self) -> Option<&mut Item> {
        match self {
            Self::Root => None,
            Self::Child(ref mut current_item) => Some(current_item),
            Self::Property(_, ref mut current_item) => Some(current_item),
        }
    }
}

impl Default for ParentRelationship {
    /// Defines a default value to be to `ItemParent::Document`.
    fn default() -> Self {
        ParentRelationship::Root
    }
}

pub trait PortableProperty:
    Default + Clone + Eq + Debug + serde::Serialize + serde::de::DeserializeOwned
{
}

impl<T: Default + Clone + Eq + Debug + serde::Serialize + serde::de::DeserializeOwned>
    PortableProperty for Option<T>
{
}

#[derive(Default, PartialEq, Eq, Debug)]
pub struct ThreadSafeProperty<T>(Arc<RefCell<T>>)
where
    T: Default + Clone + Eq + Debug + serde::Serialize + serde::de::DeserializeOwned;

fn is_option_none<T>(v: &ThreadSafeProperty<Option<T>>) -> bool
where
    T: Default + Clone + Eq + Debug + serde::Serialize + serde::de::DeserializeOwned,
{
    v.0.borrow().is_none()
}

impl<T> Clone for ThreadSafeProperty<T>
where
    T: Default + Clone + Eq + Debug + serde::Serialize + serde::de::DeserializeOwned,
{
    fn clone(&self) -> Self {
        Self(Arc::clone(&self.0))
    }
}

impl<T> ThreadSafeProperty<T>
where
    T: Default + Clone + Eq + Debug + serde::Serialize + serde::de::DeserializeOwned,
{
    pub(crate) fn new(v: T) -> ThreadSafeProperty<T> {
        Self(Arc::new(RefCell::new(v)))
    }
}

impl<T> serde::Serialize for ThreadSafeProperty<T>
where
    T: Default + Clone + Eq + Debug + serde::Serialize + serde::de::DeserializeOwned,
{
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: Serializer,
    {
        self.0.serialize(serializer)
    }
}

impl<'de, T> serde::Deserialize<'de> for ThreadSafeProperty<T>
where
    T: Default + Clone + Eq + Debug + serde::Serialize + serde::de::DeserializeOwned,
{
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: Deserializer<'de>,
    {
        T::deserialize(deserializer).map(|v| Self(Arc::new(RefCell::new(v))))
    }
}

impl<T> From<T> for ThreadSafeProperty<T>
where
    T: Default + Clone + Eq + Debug + serde::Serialize + serde::de::DeserializeOwned,
{
    fn from(value: T) -> Self {
        Self(Arc::new(RefCell::new(value)))
    }
}

/// Represents the structured form of an 'object' in Microformats.
#[derive(serde::Serialize, serde::Deserialize, Default, Debug, PartialEq, Eq, Clone)]
#[serde(rename_all = "kebab-case")]
pub struct Item {
    pub r#type: Vec<Class>,

    /// Represents the directly associated attributes for this item.
    #[serde(with = "referenced_properties")]
    pub(crate) properties: Properties,

    /// Represents a list of children for this item.
    #[serde(
        default,
        with = "referenced_children",
        skip_serializing_if = "referenced_children::is_empty"
    )]
    pub(crate) children: Arc<RefCell<Vec<Item>>>,

    /// The ID string of this item, if any is resolved.
    #[serde(default, skip_serializing_if = "is_option_none")]
    pub id: ThreadSafeProperty<Option<String>>,

    /// Represents the precise value of this item (if it's defined as a property to another).
    #[serde(default, skip_serializing_if = "is_option_none")]
    pub value: ThreadSafeProperty<Option<ValueKind>>,

    #[serde(skip)]
    parent: Arc<RefCell<ParentRelationship>>,

    #[serde(default, skip)]
    xpath: ThreadSafeProperty<String>,
}

impl PartialOrd for Item {
    fn partial_cmp(&self, other: &Self) -> Option<std::cmp::Ordering> {
        match self.id().partial_cmp(&other.id()) {
            Some(core::cmp::Ordering::Equal) => {}
            ord => return ord,
        }
        match self.value().partial_cmp(&other.value()) {
            Some(core::cmp::Ordering::Equal) => {}
            ord => return ord,
        }

        self.parent.partial_cmp(&other.parent)
    }
}

impl Ord for Item {
    fn cmp(&self, other: &Self) -> std::cmp::Ordering {
        let id = self.id().cmp(&other.id());
        let value = self.value().cmp(&other.value());

        if id != Ordering::Equal {
            id
        } else if value != Ordering::Equal {
            value
        } else {
            Ordering::Equal
        }
    }
}

impl TryFrom<serde_json::Map<String, serde_json::Value>> for Item {
    type Error = crate::Error;

    fn try_from(obj: serde_json::Map<String, serde_json::Value>) -> Result<Self, Self::Error> {
        if !obj.contains_key("type") {
            return Err(Self::Error::ObjectMissingProperty("type".to_string()));
        }
        if !obj.contains_key("properties") {
            return Err(Self::Error::ObjectMissingProperty("properties".to_string()));
        }

        serde_json::from_value(serde_json::Value::Object(obj)).map_err(Self::Error::JSON)
    }
}

impl TryFrom<serde_json::Value> for Item {
    type Error = crate::Error;

    fn try_from(v: serde_json::Value) -> Result<Self, Self::Error> {
        if let serde_json::Value::Object(o) = v {
            Self::try_from(o)
        } else {
            Err(Self::Error::NotAnObject)
        }
    }
}

impl TryInto<serde_json::Value> for Item {
    type Error = crate::Error;

    fn try_into(self) -> Result<serde_json::Value, Self::Error> {
        serde_json::to_value(self).map_err(crate::Error::JSON)
    }
}

impl IntoIterator for Item {
    type Item = Item;
    type IntoIter = std::vec::IntoIter<Self::Item>;

    fn into_iter(self) -> Self::IntoIter {
        let mut items = self
            .children()
            .into_iter()
            .flat_map(|i| i.into_iter())
            .collect::<Vec<Item>>();
        items.push(self);
        items.into_iter()
    }
}

#[derive(serde::Serialize, serde::Deserialize, Clone, Debug, PartialEq, Eq, PartialOrd, Ord)]
#[serde(untagged, rename_all = "kebab-case")]
pub enum ValueKind {
    Url(Url),
    Plain(String),
}

impl Default for ValueKind {
    fn default() -> Self {
        Self::Plain(String::default())
    }
}

mod referenced_children {

    use super::*;
    type Value = Arc<RefCell<Vec<Item>>>;

    struct ChildrenVisitor;

    impl<'de> Visitor<'de> for ChildrenVisitor {
        type Value = Value;
        fn expecting(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
            formatter.write_str("expecting a list of children nodes, an empty list or null")
        }

        fn visit_seq<ChildrenSequenceAccessor>(
            self,
            mut seq: ChildrenSequenceAccessor,
        ) -> Result<Self::Value, ChildrenSequenceAccessor::Error>
        where
            ChildrenSequenceAccessor: de::SeqAccess<'de>,
        {
            let size_hint = seq.size_hint().unwrap_or(0);
            let mut children = Vec::with_capacity(size_hint);

            while let Some(item) = seq.next_element()? {
                log::trace!("converting to item {:#?}", item);
                children.push(item);
            }

            Ok(Arc::new(RefCell::new(children)))
        }
    }

    #[allow(clippy::ptr_arg)]
    pub fn serialize<S>(children: &Value, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::ser::Serializer,
    {
        let mut seq = serializer.serialize_seq(Some(children.borrow().deref().len()))?;
        let safe_items = children
            .borrow()
            .deref()
            .iter()
            .filter(|item| !item.is_empty())
            .map(|item| item.deref().clone())
            .collect::<Vec<_>>();
        for concrete_item in safe_items {
            seq.serialize_element(&concrete_item)?;
        }
        seq.end()
    }

    pub fn deserialize<'de, D>(deserializer: D) -> Result<Value, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        deserializer.deserialize_seq(ChildrenVisitor)
    }

    pub fn is_empty(items: &Arc<RefCell<Vec<Item>>>) -> bool {
        items.borrow().deref().is_empty()
    }
}

mod referenced_properties {
    use super::*;
    type Value = Properties;

    struct PropertyVisitor;

    #[derive(serde::Deserialize, Debug)]
    #[serde(untagged)]
    enum PotentialValues {
        List(NodeList),
        Value(PropertyValue),
    }

    impl<'de> Visitor<'de> for PropertyVisitor {
        type Value = Value;

        fn expecting(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
            formatter.write_str("a map of properties with values that could be null, a string, a list of either strings, maps or both")
        }

        fn visit_map<A>(self, mut map_visitor: A) -> Result<Self::Value, A::Error>
        where
            A: de::MapAccess<'de>,
        {
            let property_map = Properties::default();
            *property_map.0.deref().borrow_mut() =
                HashMap::with_capacity(map_visitor.size_hint().unwrap_or(0));

            while let Some(key) = map_visitor.next_key()? {
                let concrete_value: NodeList = match map_visitor.next_value::<PotentialValues>()? {
                    PotentialValues::List(values) => values,
                    PotentialValues::Value(node) => vec![node],
                };

                // FIXME: Use `Properties:try_insert` to prevent overwriting of values; merge them.
                property_map
                    .0
                    .deref()
                    .borrow_mut()
                    .insert(key, concrete_value);
            }

            Ok(property_map)
        }
    }

    pub fn serialize<S>(properties: &Value, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::ser::Serializer,
    {
        let mut properties_seq = serializer.serialize_map(Some(properties.0.borrow().len()))?;

        for (key, value) in properties.0.borrow().iter() {
            properties_seq.serialize_entry(key, value)?;
        }

        properties_seq.end()
    }

    pub fn deserialize<'de, D>(deserializer: D) -> Result<Value, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        deserializer.deserialize_map(PropertyVisitor)
    }
}

impl Item {
    /// Provides the parent item of this item.
    pub fn parent(&self) -> ParentRelationship {
        self.parent.deref().borrow().to_owned()
    }

    /// Provides a copy of the properties.
    pub fn properties(&self) -> HashMap<String, Vec<PropertyValue>> {
        self.properties.0.borrow().deref().to_owned()
    }

    /// Returns this Item without a reference to the document.
    pub fn into_dangling(&self) -> Self {
        let mut dangling_item = self.clone();
        dangling_item.set_parent(ParentRelationship::Root);
        dangling_item
    }

    /// Creates a new item with the provided `ItemParent` as its parent.
    pub fn new(parent: ParentRelationship, types: Vec<Class>) -> Self {
        Item {
            r#type: types,
            parent: Arc::new(RefCell::new(parent)),
            ..Default::default()
        }
    }

    /// Adds a Item to this Item as a child.
    pub fn append_child(&mut self, item: &Self) {
        self.children.deref().borrow_mut().push(item.clone());
    }

    /// Determines if this item is undefined - an empty one.
    pub fn is_empty(&self) -> bool {
        self.children().is_empty() && self.r#type.is_empty()
    }

    pub fn remove_child(&mut self, problem_child: &Self) {
        if let Some(position) = self.children().iter().position(|v| v == problem_child) {
            trace!(
                "Attempting to remove the child {:#?} from {:#?}",
                problem_child,
                self.children
            );
            self.children.deref().borrow_mut().remove(position);
        }
    }

    pub fn remove_whole_property(&mut self, property_name: &str) {
        let mut props = self.properties.0.deref().borrow_mut();
        trace!(
            "Removed {:?} from this item's properties; which contained {:#?}",
            property_name,
            props.remove(property_name)
        );
    }

    pub fn content(&self) -> Option<Vec<PropertyValue>> {
        self.properties.0.borrow().deref().get("content").cloned()
    }

    pub fn append_property(&mut self, property_name: &str, property_value: PropertyValue) {
        let mut props = self.properties.0.deref().borrow_mut();
        let mut new_values = if let Some(values) = props.get(property_name) {
            values.to_vec()
        } else {
            Vec::default()
        };

        trace!(
            "Appending the value {:?} to the {:?} property",
            property_value,
            property_name
        );
        new_values.push(property_value);

        props.insert(property_name.to_owned(), new_values);
    }

    /// Checks if this item has any properties with a nested item.
    pub fn has_nested_microformats(&self) -> bool {
        let props = self.properties.0.borrow();
        let has_nested_value_microformats = props
            .values()
            .flatten()
            .any(|v| matches!(v, PropertyValue::Item(_)));

        has_nested_value_microformats || !self.children().is_empty()
    }

    /// Obtains a list of the immediate children.
    pub fn children(&self) -> Vec<Item> {
        self.children.borrow().deref().to_owned()
    }

    pub fn nested_children(&self) -> Vec<Item> {
        self.properties()
            .values()
            .flatten()
            .filter_map(|value| {
                if let PropertyValue::Item(item) = value {
                    Some(item)
                } else {
                    None
                }
            })
            .cloned()
            .collect::<Vec<_>>()
    }

    pub fn has_id(&self, id: &str) -> Option<Item> {
        self.id()
            .and_then(|my_id| {
                if my_id == id {
                    Some(self.clone())
                } else {
                    None
                }
            })
            .or_else(|| {
                self.children()
                    .iter()
                    .find_map(|item| Self::has_id(item, id))
            })
            .or_else(|| {
                self.nested_children()
                    .iter()
                    .find_map(|item| Self::has_id(item, id))
            })
    }

    pub fn has_url(&self, url: &Url) -> Option<PropertyValue> {
        self.properties()
            .get("url")
            .and_then(|urls| {
                if urls.contains(&PropertyValue::Url(url.to_owned())) {
                    Some(PropertyValue::Item(self.clone()))
                } else {
                    None
                }
            })
            .or_else(|| self.children().iter().find_map(|item| item.has_url(url)))
            .or_else(|| {
                self.nested_children()
                    .iter()
                    .find_map(|item| item.has_url(url))
            })
    }

    pub fn set_parent(&mut self, property_relationship: ParentRelationship) {
        *self.parent.borrow_mut().deref_mut() = property_relationship;
    }

    pub fn properties_with_matching_value(&self, property_value: PropertyValue) -> Vec<String> {
        let self_properties: Vec<_> = self
            .properties()
            .iter()
            .filter_map(|(property_name, property_values)| {
                property_values.iter().find_map(|value| {
                    let pv = property_value.clone();
                    if *value == pv {
                        return Some(property_name.to_owned());
                    }

                    debug!("Comparing {:?} to {:?}", pv, value);

                    match value {
                        PropertyValue::Plain(p) => {
                            if let PropertyValue::Url(u) = &pv {
                                if p.contains(&u.to_string()) {
                                    Some(property_name.to_owned())
                                } else {
                                    None
                                }
                            } else {
                                None
                            }
                        }
                        PropertyValue::Url(u) => {
                            if pv == PropertyValue::Plain(u.to_string()) {
                                Some(property_name.to_owned())
                            } else {
                                None
                            }
                        }
                        PropertyValue::Temporal(_) => None,
                        PropertyValue::Fragment(Fragment { html, value, .. }) => {
                            if let PropertyValue::Url(u) = &pv {
                                if html.contains(&u.to_string()) || value.contains(&u.to_string()) {
                                    Some(property_name.to_owned())
                                } else {
                                    None
                                }
                            } else if let PropertyValue::Plain(p) = &pv {
                                if html.contains(p) || value.contains(p) {
                                    Some(property_name.to_owned())
                                } else {
                                    None
                                }
                            } else {
                                None
                            }
                        }
                        PropertyValue::Image(Image { src, alt }) => {
                            if let PropertyValue::Url(u) = &pv {
                                if src == u {
                                    Some(property_name.to_owned())
                                } else {
                                    None
                                }
                            } else if let PropertyValue::Plain(t) = &pv {
                                if alt == t {
                                    Some(property_name.to_owned())
                                } else {
                                    None
                                }
                            } else {
                                None
                            }
                        }
                        PropertyValue::Item(item) => {
                            if let PropertyValue::Url(u) = &pv {
                                let props = item.properties().clone();

                                // FIXME: Check fragment of URL if self-page.
                                let urns = [
                                    props.get("url").cloned().unwrap_or_default(),
                                    props.get("uid").cloned().unwrap_or_default(),
                                ]
                                .iter()
                                .flatten()
                                .cloned()
                                .collect::<Vec<_>>();

                                if let Some(ref v) = item.value() {
                                    match v {
                                        ValueKind::Url(vu) if vu == u => {
                                            Some(property_name.to_owned())
                                        }
                                        ValueKind::Plain(vp) if *vp == u.to_string() => {
                                            Some(property_name.to_owned())
                                        }
                                        _ => None,
                                    }
                                } else if urns.iter().any(|v| *v == PropertyValue::Url(u.clone())) {
                                    Some(property_name.to_owned())
                                } else {
                                    None
                                }
                            } else {
                                None
                            }
                        }
                    }
                })
            })
            .collect();

        let children_properties = self
            .children()
            .iter()
            .flat_map(|item| {
                item.properties_with_matching_value(property_value.clone())
                    .into_iter()
            })
            .collect::<Vec<_>>();

        let mut items = [self_properties, children_properties]
            .iter()
            .flatten()
            .cloned()
            .collect::<Vec<_>>();

        items.sort();
        items.dedup();
        items
    }

    /// Resolve full date time values for those missing them.
    /// This is an implementation of <https://microformats.org/wiki/value-class-pattern#microformats2_parsers_implied_date>.
    pub fn concatenate_times(&mut self) {
        trace!("Attempting to resolve the implied dates for time values.");

        let mut last_seen_date: Option<temporal::Date> = None;

        self.properties_mut()
            .values_mut()
            .flat_map(|properties| properties.iter_mut())
            .for_each(|property| {
                if let PropertyValue::Temporal(temporal::Value::Timestamp(timestamp)) = property {
                    if timestamp.date.is_some() {
                        trace!("Recording {:?} as the last seen date.", timestamp.date);
                        last_seen_date = timestamp.date.clone();
                    } else {
                        trace!(
                            "Using {:?} as the date with {:?}",
                            last_seen_date,
                            timestamp.date
                        );
                        timestamp.date = last_seen_date.clone();
                        last_seen_date = None;
                    }
                }
            })
    }

    /// Obtains a list of values for a property.
    pub fn get_property(&self, arg: &str) -> Option<Vec<PropertyValue>> {
        self.properties.0.borrow().deref().get(arg).cloned()
    }

    pub fn set_id(&mut self, id_value: Option<String>) {
        trace!("Set the ID of this element to be {:?}", id_value);
        *self.id.0.deref().borrow_mut() = id_value;
    }

    pub fn set_value(&mut self, value: Option<ValueKind>) {
        trace!("Set the value of this element to be {:?}", value);
        *self.value.0.deref().borrow_mut() = value;
    }

    pub fn value(&self) -> Option<ValueKind> {
        self.value.0.borrow().deref().to_owned()
    }

    pub fn id(&self) -> Option<String> {
        self.id.0.borrow().deref().to_owned()
    }

    pub fn properties_mut(&self) -> RefMut<'_, HashMap<String, Vec<PropertyValue>>> {
        self.properties.0.deref().borrow_mut()
    }

    pub(crate) fn children_mut(&self) -> RefMut<'_, Vec<Item>> {
        self.children.deref().borrow_mut()
    }

    fn reassociate_parents(&mut self) {
        self.children_mut().iter_mut().for_each(|child| {
            child.set_parent(ParentRelationship::Child(self.clone()));
        });

        for (property_name, property_values) in self.properties_mut().iter_mut() {
            for property_value in property_values.iter_mut() {
                if let PropertyValue::Item(item) = property_value {
                    if item.get_property("url").is_some() {
                        item.set_parent(ParentRelationship::Property(
                            RelationshipKind::URL(property_name.to_string()),
                            self.clone(),
                        ))
                    } else {
                        item.set_parent(ParentRelationship::Property(
                            RelationshipKind::HTML(property_name.to_string()),
                            self.clone(),
                        ))
                    }
                }
            }
        }
    }
}

#[derive(
    Clone, Debug, PartialEq, Eq, PartialOrd, Ord, Default, serde::Serialize, serde::Deserialize,
)]
pub struct Relation {
    pub rels: Vec<String>,

    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub hreflang: Option<String>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub media: Option<String>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub title: Option<String>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub r#type: Option<String>,
    #[serde(default, skip_serializing_if = "Option::is_none")]
    pub text: Option<String>,
}

impl Relation {
    /// Fuses the values of the other relation with this one.
    pub fn merge_with(&mut self, other: Self) {
        self.rels.extend_from_slice(&other.rels);
        self.rels.sort();
        self.rels.dedup();

        if self.hreflang.is_none() {
            self.hreflang = other.hreflang;
        }

        if self.media.is_none() {
            self.media = other.media;
        }
        if self.title.is_none() {
            self.title = other.title;
        }
        if self.r#type.is_none() {
            self.r#type = other.r#type;
        }
        if self.text.is_none() {
            self.text = other.text;
        }
    }
}

#[derive(Clone, Debug, PartialEq, Eq, Default, serde::Deserialize, serde::Serialize)]
pub struct Relations {
    #[serde(flatten)]
    pub items: HashMap<Url, Relation>,
}

impl Relations {
    pub fn by_rels(&self) -> HashMap<String, Vec<Url>> {
        let mut rels: HashMap<String, Vec<Url>> = HashMap::default();
        self.items
            .iter()
            .flat_map(|(u, rel)| {
                rel.rels
                    .iter()
                    .map(move |rel_name| (rel_name.to_owned(), u.to_owned()))
            })
            .for_each(|(rel_name, url)| {
                if let Some(rel_urls) = rels.get_mut(&rel_name) {
                    rel_urls.push(url);
                } else {
                    rels.insert(rel_name, vec![url]);
                }
            });

        rels.iter_mut().for_each(|(_, urls)| {
            urls.dedup();
            urls.sort()
        });

        rels
    }
}

/// Represents a parsed document of Microformats items and its relating rel links.
// The long-term goal of this is to have the items be a hashmap of all of the
// resolved items by a particular hashable ID. Then for each Item, we have a
// ItemID that can be used to look up the values to reduce the number of unbounded
// pointers made to a central value. It also make the resolving of a particular
// item in a whole page faster and even possible to do anchor-specific resolving
// of items in a document _without_ a provided URL.
// FIXME: Implement serde::Deserialize to build map of items by internal IDs against tree of items.
// FIXME: Re-build the tree for serialization.
#[derive(Clone, Debug, PartialEq, Default, Eq)]
pub struct Document {
    items: ThreadSafeProperty<Vec<Item>>,
    pub url: Option<url::Url>,
    pub rels: Relations,
    pub lang: Option<String>,
}

impl Document {
    pub fn new(url: Option<Url>) -> Arc<RefCell<Self>> {
        Arc::new(RefCell::new(Self {
            url,
            ..Default::default()
        }))
    }

    pub fn items(&self) -> Vec<Item> {
        self.items.0.borrow().deref().clone()
    }

    /// Creates a new top-level child item for this document.
    pub fn create_child_item(&mut self, types: &[Class]) -> Item {
        let item = Item::new(ParentRelationship::Root, types.to_vec());
        self.items.0.deref().borrow_mut().push(item.to_owned());
        item
    }

    /// Adds a new [relation][Relation] pointing to the [url][Url] provided.
    pub fn add_relation(&mut self, url: Url, relation: Relation) {
        if let Some(rel) = self.rels.items.get_mut(&url) {
            rel.merge_with(relation);
        } else {
            self.rels.items.insert(url.to_owned(), relation);
        }
    }

    /// Finds the property in this document that's preferred by this URL.
    // FIXME: This doesn't go deep enough, implement transverser.
    pub fn get_item_by_url(&self, url: &Url) -> Option<PropertyValue> {
        self.items().iter().find_map(|item| item.has_url(url))
    }

    /// Finds the property in this document that has the provided ID.
    // FIXME: This doesn't go deep enough, implement transverser.
    pub fn get_item_by_id(&self, id: &str) -> Option<Item> {
        self.items().iter().find_map(|item| item.has_id(id))
    }

    /// Finds the item and property name that this exists in.
    // FIXME: Implement transverser.
    pub fn find_property_name_with_value(
        &self,
        property_value: PropertyValue,
    ) -> Vec<(Item, String)> {
        let mut items = self
            .items()
            .iter()
            .cloned()
            .flat_map(|item| {
                let self_item = item.to_owned();
                item.properties_with_matching_value(property_value.clone())
                    .into_iter()
                    .map(move |name| (self_item.to_owned(), name))
            })
            .collect::<Vec<_>>();

        items.sort();
        items.dedup();
        items
    }
}

impl serde::Serialize for Document {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        let mut s = serializer.serialize_struct("Document", 3)?;

        s.serialize_field("items", &self.items)?;
        s.serialize_field("rel-urls", &self.rels.items)?;
        s.serialize_field("rels", &self.rels.by_rels())?;
        if let Some(lang) = &self.lang {
            s.serialize_field("lang", lang)?;
        }
        s.end()
    }
}

#[derive(serde::Deserialize, Debug)]
#[serde(field_identifier, rename_all = "kebab-case")]
enum DocumentDeserializationFields {
    Items,
    RelUrls,
    Rels,
    Url,
    Lang,
}

impl<'de> serde::Deserialize<'de> for Document {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        struct DocumentVisitor;

        impl<'de> Visitor<'de> for DocumentVisitor {
            type Value = Document;

            fn expecting(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
                formatter.write_str("a Microformat document represented with the expected fields")
            }

            fn visit_map<V>(self, mut map: V) -> Result<Document, V::Error>
            where
                V: de::MapAccess<'de>,
            {
                let mut document = Document::default();
                let mut rel_urls: Option<Relations> = None;

                while let Ok(Some(key)) = map.next_key() {
                    match key {
                        DocumentDeserializationFields::Items => {
                            let raw_items = map.next_value::<Vec<Item>>()?;
                            let items = raw_items
                                .into_iter()
                                .map(|mut item| {
                                    item.set_parent(ParentRelationship::Root);
                                    item.reassociate_parents();
                                    item
                                })
                                .collect::<Vec<_>>();

                            document.items.0.deref().borrow_mut().extend(items);
                        }
                        DocumentDeserializationFields::Url => {
                            if document.url.is_some() {
                                return Err(de::Error::duplicate_field("url"));
                            }

                            document.url = map.next_value()?;
                        }
                        DocumentDeserializationFields::RelUrls => {
                            if rel_urls.is_some() {
                                return Err(de::Error::duplicate_field("rel-urls"));
                            }

                            rel_urls = map.next_value()?;
                        }
                        DocumentDeserializationFields::Lang => {
                            if document.lang.is_some() {
                                return Err(de::Error::duplicate_field("lang"));
                            }

                            document.lang = map.next_value()?;
                        }
                        DocumentDeserializationFields::Rels => {
                            map.next_value::<HashMap<String, Vec<String>>>()?;
                        }
                    }
                }

                document.rels = rel_urls.unwrap_or_default();

                Ok(document)
            }
        }

        deserializer.deserialize_struct(
            "Document",
            &["items", "rel-urls", "url", "lang"],
            DocumentVisitor,
        )
    }
}

impl IntoIterator for Document {
    type Item = Item;
    type IntoIter = std::vec::IntoIter<Self::Item>;

    fn into_iter(self) -> Self::IntoIter {
        self.items()
            .into_iter()
            .flat_map(|i| i.into_iter())
            .collect::<Vec<Item>>()
            .into_iter()
    }
}

impl From<Document> for ParentRelationship {
    fn from(_: Document) -> Self {
        ParentRelationship::Root
    }
}
