///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================

/*Prog:csr
NAME: @code{csr} - sparse matrix manipulation 
@pindex csr
@pindex makefish
@toindex @code{octave}
@toindex @code{matlab}
@toindex @code{ghostview}
@fiindex @file{.hb} Harwell-Boeing matrix
@fiindex @file{.mm} Matrix-Market matrix
@fiindex @file{.m} matlab matrix
@fiindex @file{.ps} postscript
SYNOPSIS:
  @example
   csr @var{options...}
  @end example
DESCRIPTION:       
  Plots or manipulates a sparse matrix.
  Read from standard input, in Harwell-Boeing or Matrix-Market format.
EXAMPLE: 
  @noindent
  Enter commands as:
  @example
    	csr -ps < matrix.hb | ghostview -
  @end example
  @noindent
  or, for small one, in matlab form:
  @example
    	csr -matlab < small.hb
  @end example

INPUT FORMAT OPTIONS:
  @table @code
    @itemx -input-hb
    	input in Harwell-Boeing format. This is the default.
    @itemx -input-mm
    	input in Matrix-Market format.
  @end table
OUTPUT FORMAT OPTIONS:
  @table @code
    @itemx -hb
    	output in Harwell-Boeing format. This is the default.
    @itemx -mm
    	output in Matrix-Market format.
    @itemx -ps
    	output postscript. This is the default.
    @itemx -ml
    @itemx -matlab
    	output matlab format (see octave(1)).
       Be carreful for large matrix. Use instead the sparse matlab option.
    @itemx -sparse-matlab
    	output matlab sparse format (see octave(1)).
    @itemx -name @var{string}
    	specifies a current name for the sparse matlab output option.
    @itemx -dump
    	dump matrix on standard output.
  @end table
GRAPHIC OPTIONS:
  @table @code
    @itemx -black-and-white
    	plot color in black and white style. Non-null
	entries appears in black.
    @itemx -color
    	plot non-null matrix entries in color. this is the default.
    @itemx -logscale
    	map matrix entries in color using a logscale mode of the 
	 absolute value. Null entries appear in white. This 
	 option is valid when the @code{-ps} and @code{-color}
	 options are active.
    @itemx -nologscale
    	this is the default.
  @end table

OPERATIONS:
  @table @code
    @itemx -tril [@var{k}]
    	output the lower triangular part.
      The optional argument k specifies how many diagonals above the main diagonal should
      also be set to zero. If the value of k is negative, additional elements above
      are also selected.
    @itemx -triu [@var{k}]
    	output the upper triangular part.
      The optional argument k specifies how many diagonals below the main diagonal should
      also be set to zero. If the value of k is negative, additional elements below
      are also selected.
    @itemx -gibbs
    	output reordered matrix using the Gibbs-King profile reduction algorithm.
  @end table

TRANPOSITION:
  @table @code
    @itemx -notranspose-in
    	The Harwell-Boeing format uses a column-oriented (CSC) matrix description.
      This command uses the row-oriented (CSR) matrix description.
      Since the direct interpretation from CSC to CSR is equivalent to a transposition,
      we get with this option the transposed matrix.

      The transpose option allow a direct interpretation of the
      CSR matrix input format in CSC, as used by HB files.

    @itemx -transpose-in
    	Convert input from CSC to CSR. This is the default.

    @itemx -notranspose-out
    	The same for output stream

    @itemx -transpose-out
    	The same for output stream
  @end table


RIGHT-HAND SIDE OPTION:
  @table @code
    @itemx -rhs
      Get also right-hand side in the H.B. input format.
      For more details, see related H.B. format documentation.
      Thus, force also @code{-hb} output format option.
      This option is incompatible with options operating on matrix
  @end table
POSTSCRIPT FORMAT:
  @example
    	makefish 10 | csr -ps | ghostview -
  @end example
  The @code{makefish} command generates the finite-difference
  discretised bidimensional Poisson operator in Harwell-Boeing format. 
  This matrix is piped to @code{csr} and then to @code{ghostview}
  for visualisation.
  Only color plot output is yet implemented.
  Gray-scale and black-and white will be soon too.

MATLAB FORMAT:
  @example
    	makefish 4 | csr -matlab > fish4.m
  @end example
  Instead of matlab, you can use @code{octave} program.
  Octave is matlab-compatible.
  Octave is intended for numerical computations.
  It provides a convenient command line interface for
  solving linear and nonlinear problems numerically. 
  Octave has been developped by John W. Eaton 
  @email{jwe@@che.utexas.edu}, Department of Chemical Engineering,
  The University of Texas at Austin.
  Octave is free software; see 
  @url{http://www.che.wisc.edu/octave}
  for more information.

DUMP FORMAT:
  This format is suitable for small matrix.
  The matrix is printed to screen in a dense format.
  @example
	makefish 4 | csr -dump
  @end example

HARWELL-BOEING FORMAT:       
  @noindent
  This is the standard matrix file format.
  For a complete description of this format,
  see @cite{Duff, et al.,  ACM TOMS Vol.15, No.1, March 1989}, or
  @quotation
    @url{http://www.psc.edu/general/software/packages/hbsmc/manual/toc.html}
  @end quotation
  or the @cite{CERFACS user's guide report} available via 
  anonymous ftp on
  @quotation
    @url{ftp://ftp.cerfacs.fr/pub/harwell_boeing/}.
  @end quotation
   
  @noindent
  Most of the features are available for 
  Harwell-Boeing format,
  but some things remains to do.
  Here is some limitations on the input matrix format type:
  @table @code
      @itemx C..
      	Complex values
      @itemx M..
      	Sparse right-hand side
  @end table
  @noindent
  and @code{KEY} options that remains to do:
  @table @code
      @itemx .S.
      	need to be symmetrized (may be)
      @itemx .Z.
      	need to be anti-symmetrized (may be)
      @itemx R.E
      	Elemented Real (could be)
  @end table
AUTHOR: 
    LMC-IMAG, 38041 Grenoble cedex 9, France
    | Pierre.Saramito@imag.fr
SEE ALSO:
    "iorheo"(3), "octave"(1)
DATE:
    12 may 1997
End:
*/

#include "rheolef/skit.h"
using namespace rheolef;
using namespace std;

void usage()
{
      cerr << "csr: usage: csr "
	   << "[-hb|-mm|-ps|-ml|-matlab|-sparse-matlab|-dump] "
	   << "[-name string] "
	   << "[-input-hb|-input-mm] "
	   << "[-color|-black-and-white] "
	   << "[-[no]logscale] "
	   << "[-gibbs] "
	   << "[-tril [k]] "
	   << "[-triu [k]] "
	   << "[-[no]transpose-[in|out]] "
	   << "[-rhs] "
	   << endl;
      exit (1);
}
int main(int argc, char **argv)
{
    if (argc <= 1) usage();
    //
    // set default options
    //
    cout << hb << setbasename("a");

    bool get_rhs  = false;

    //
    // get input options
    //
    for (int i = 1; i < argc; i++) {
	     if (strcmp (argv[i], "-transpose-in") == 0)    cin >> transpose;
	else if (strcmp (argv[i], "-notranspose-in") == 0)  cin >> notranspose;
        else if (strcmp (argv[i], "-input-hb") == 0)        cin >> hb;
        else if (strcmp (argv[i], "-input-mm") == 0)        cin >> matrix_market;
    }
    //
    // input matrix
    //
    csr<Float> a;
    cin >> a;
    //
    // get options
    //
    for (int i = 1; i < argc; i++) {
      
             if (strcmp (argv[i], "-hb") == 0)              cout << hb;
        else if (strcmp (argv[i], "-mm") == 0)              cout << matrix_market;
        else if (strcmp (argv[i], "-ml") == 0)              cout << ml;
        else if (strcmp (argv[i], "-matlab") == 0)          cout << matlab;
        else if (strcmp (argv[i], "-sparse-matlab") == 0)   cout << sparse_matlab;
        else if (strcmp (argv[i], "-ps") == 0)              cout << ps;
        else if (strcmp (argv[i], "-dump") == 0)            cout << dump;

	else if (strcmp (argv[i], "-tril") == 0 ||
		 strcmp (argv[i], "-triu") == 0)  {
	    int k = 0;
	    int io = i;
	    if (i+1 < argc) {
	        if (isdigit(argv[i+1][0]))
		    k = atoi(argv[++i]);
                else if (argv[i+1][0] == '-' && 
		    isdigit(argv[i+1][1]))
		    k = - atoi(argv[++i]+1);
            }
	    if (strcmp (argv[io], "-tril") == 0)            a = tril(a,k);
            else                                            a = triu(a,k);
        }
	else if (strcmp (argv[i], "-gibbs") == 0) {
	    						    permutation p = gibbs(a);
	    						    a = perm(a, p, p);   
	}
        else if (strcmp (argv[i], "-rhs") == 0) { 
							    cout << hb; 
							    get_rhs = true; 
	}
        else if (strcmp (argv[i], "-logscale") == 0)        cout << logscale;
        else if (strcmp (argv[i], "-nologscale") == 0)      cout << nologscale;
        else if (strcmp (argv[i], "-color") == 0)           cout << color;
        else if (strcmp (argv[i], "-black-and-white") == 0) cout << black_and_white;

	else if (strcmp (argv[i], "-transpose-out") == 0)   cout << transpose;
	else if (strcmp (argv[i], "-notranspose-out") == 0) cout << notranspose;
	else if (strcmp (argv[i], "-name") == 0) {
		if (++i >= argc) {
		    cerr << "-name: missing argument" << endl; 
		    usage();
		}
		cout << setbasename (argv[i]);
	}
	// skit input options
	else if (strcmp (argv[i], "-transpose-in") == 0)    ;
	else if (strcmp (argv[i], "-notranspose-in") == 0)  ;
        else if (strcmp (argv[i], "-input-hb") == 0)        ;
        else if (strcmp (argv[i], "-input-mm") == 0)        ;

        else {
	    cerr << "unexpected command-line argument `" << argv[i] << "'" << endl; 
	    usage();
        }
    }
    //
    // output matrix
    //
    if (!get_rhs) {
    
	// simple output
	cout << a;

    } else {
    
	unsigned int n = iorheo::getnrhs(cin);

	//      Here, we get the number of right-hand-sides.
	//      Next, get the type: has guess and/or exact solution:

        string rhs_type = iorheo::getrhstype(cin);
    
        //      and start to write.
        //      We specify first in header the number of right-hand-side,
        //      and then output the matrix.
    
        cout << hb << setnrhs(n) << setrhstype(rhs_type) << notranspose << a ;
    
        //      A first loop for optional rhs:

        for (unsigned int i = 0; i < n; i++) {
            vec<Float> b;
            cin  >> b;
            cout << b;
        }
        //
        //      Test if guess vectors is supplied:
        //
        if (rhs_type[1] == 'G') {
            vec<Float> x0;
            for (unsigned int i = 0; i < n; i++) {
                cin  >> x0;
                cout << x0;
            }
        }
        //
        //	    and finally, check if an exact solution vector is supplied
        //
        if (rhs_type[2] == 'X') {
            vec<Float> x;
            for (unsigned int i = 0; i < n; i++) {
                cin  >> x;
                cout << x;
            }
        }
    }
    return 0;
}
