/* V4L module for the GyachE-Broadcaster program:
   This application is used to send webcam streams to
    Yahoo. Right now this program has only been tested
    with a single Video4Linux device: An OV511 DLink
    C100 USB webcam.  The program uses Video4Linux-1
    for image capture and the libJasper library for Jpeg-2000
    image conversions.  */

/* This program borrows alot of code from both Ayttm and 
    Gyach-E itself, as well as the old 'videodog' program
    for a few decent V4L usage examples.

    It is designed for simplicity, speed, 
    memory-friendliness, and stability: It runs as an EXTERNAL 
    program to Gyach Enhanced, so that if it DOES crash, it 
    crashes ALONE, rather than taking down an entire chat program
    with it. It is a clean, efficient SINGLE-THREADED application 
*/

/* Note: This modules is coded for support for EITHER pthreads, or Gthread's
 * The Gthreads code is tested and confirmed working.
 * 
 * The pthreads code is eyeballed, and *should* work. (note that it hasn't
 * even been compiled...
 * ghosler - 20080413 - GyachI 1.1.64
 */


/*****************************************************************************
 * gyachewebcam.c
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 *
 * Copyright (C) 2003-2005 Erica Andrews (Phrozensmoke ['at'] yahoo.com)
 * Released under the terms of the GPL.
 * *NO WARRANTY*
 *****************************************************************************/
/*****************************************************************************
 * 20060118 Video4Linux Version 2 support 
 * 20060208 Video4Linux Version 1 improvements
 * 
 * Stefan Sikora
 * <hoshy['at']schrauberstube['dot']de>
 *
 * 20081020 thru 20081131 Many updates/changes
 * 20090413 code re-organization, added saving of self video frames.
 *
 * Greg Hosler
 * ghosler [at] users.sourceforge.net
 ****************************************************************************/


#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <sys/stat.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/socket.h>
#include <sys/select.h>
#include <arpa/inet.h>
#include <locale.h>
#include <ctype.h>
#include <netdb.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include <assert.h>

#include <gtk/gtk.h>
#include <glib.h>

#ifdef HAVE_LIBV4L
#include <libv4l2.h>
#include <libv4l1.h>
#else
#define v4l1_open   open
#define v4l1_close  close
#define v4l1_ioctl  ioctl
#define v4l1_read   read
#define v4l1_mmap   mmap
#define v4l1_munmap munmap

#define v4l2_open   open
#define v4l2_close  close
#define v4l2_ioctl  ioctl
#define v4l2_read   read
#define v4l2_mmap   mmap
#define v4l2_munmap munmap
#endif

/* V4L1 */
#include <linux/videodev.h>

/*  V4L2 */
#include <linux/videodev2.h>


#include "gyacheupload.h"
#include "v4l-fmtconv.h"
#include "camcapt.h"

#include "gyachi_lib.h"

#if !defined(G_THREADS_ENABLED) && defined (USE_PTHREAD_CREATE)
#include <pthread.h>
#endif

int exit_on_error=0;
int cam_is_open=0;
int width=320, height=240, bits_per_pixel=3;
unsigned char *v_device; /* device */

// Supported palette
#define PAL_JPEG    (0)
#define PAL_RGB24   (1)
#define PAL_RGB32   (2)
#define PAL_YUV420P (3)
#define PAL_YUV422P (4)
#define PAL_RGB565  (5)
#define PAL_RGB555  (6)
#define PAL_GREY    (7)
#define PAL_DEFAULT PAL_RGB24

/*  V4L2 */
struct v4l2_capability grab_cap_v4l2;
struct v4l2_format grab_fmt;

struct buffer {
        void *start;
        size_t length;
};

struct buffer *buffers = NULL;
static unsigned int n_buffers = 0;
/* end V4L2 */

/* V4L1 */
struct video_capability grab_cap_v4l1;

typedef struct {
	char    *name;
	uint16_t palette;
	uint16_t depth;
} webcam_palette_t;


webcam_palette_t wpalette[] = {
	{ "JPEG",    VIDEO_PALETTE_RGB24,   24 },
	{ "RGB24",   VIDEO_PALETTE_RGB24,   24 },
	{ "RGB32",   VIDEO_PALETTE_RGB32,   32 },
	{ "YUV420P", VIDEO_PALETTE_YUV420P, 12 },
	{ "YUV422P", VIDEO_PALETTE_YUV422P, 12 },
	{ "RGB565",  VIDEO_PALETTE_RGB565,  16 },
	{ "RGB555",  VIDEO_PALETTE_RGB555,  16 },
	{ "GREY",    VIDEO_PALETTE_GREY,    8 },
	{ "GRAY",    VIDEO_PALETTE_GREY,    8 },
	{ NULL, 0, 0 }
};

struct video_picture grab_pic;
struct video_mmap grab_buf;
struct video_mbuf grab_mbuf;
/* end  V4L1 */

/* forward declarations */
int grab_init_v4l1(GtkWidget *parent);
void cleanup_v4l1(GtkWidget *parent);
void get_cam_settings_v4l1(GtkWidget *parent);
void set_picture_v4l1(GtkWidget *parent);
unsigned char* grab_one_v4l1(int *width, int *height, GtkWidget *parent);
int grab_init_v4l2(GtkWidget *parent);
void cleanup_v4l2(GtkWidget *parent);
void get_cam_settings_v4l2(GtkWidget *parent);
void set_picture_v4l2(GtkWidget *parent);
unsigned char* grab_one_v4l2(int *width, int *height, GtkWidget *parent);
void init_pnm_buf(GtkWidget *parent);


typedef struct {
	int (*grab_init)(GtkWidget *parent);
	void (*cleanup)(GtkWidget *parent);
	void (*get_cam_settings)(GtkWidget *parent);
	void (*set_picture)(GtkWidget *parent);
	unsigned char *(*grab_one)(int *width, int *height, GtkWidget *parent);
} V4L_API;

V4L_API v4l_call[] = { {grab_init_v4l1, cleanup_v4l1, get_cam_settings_v4l1, set_picture_v4l1, grab_one_v4l1},
		       {grab_init_v4l2, cleanup_v4l2, get_cam_settings_v4l2, set_picture_v4l2, grab_one_v4l2}};
int v4l_idx = 0;
char *v4l_version = NULL;

int grab_fd=-1;
int grab_size;
int grab_palette = PAL_RGB24;
unsigned char *grab_data;
char *pnm_buf=NULL;
int pnm_size;
unsigned char *rgb_buf;

static char webcamrc_path[256];

V4L_PROPERTY cam_properties[] = {
	{.name="hue",			.v4l2_CID=V4L2_CID_HUE,			.toggle=0 },
	{.name="contrast",		.v4l2_CID=V4L2_CID_CONTRAST,		.toggle=0 },
	{.name="brightness",		.v4l2_CID=V4L2_CID_BRIGHTNESS,		.toggle=0 },
	{.name="saturation",		.v4l2_CID=V4L2_CID_SATURATION,		.toggle=0 },
	{.name="gamma",			.v4l2_CID=V4L2_CID_GAMMA,		.toggle=0 },
	{.name=NULL,			.v4l2_CID=0 }
};
GList *cam_property_list=NULL;

void	free_cam_property_list() {
	GList *cam_property_entry;
	V4L_PROPERTY *property;

	for (cam_property_entry = cam_property_list;
	     cam_property_entry;
	     cam_property_entry = cam_property_entry->next) {
		property = (V4L_PROPERTY *)cam_property_entry->data;
		if (property->name) free(property->name);
		free(property);
	}
	g_list_free(cam_property_list);
	cam_property_list = NULL;
}

void	initialize_cam_property_list() {
	V4L_PROPERTY *cam_property;
	V4L_PROPERTY *new_property;

	if (cam_property_list) {
		free_cam_property_list();
	}

	for (cam_property = cam_properties;
	     cam_property->name;
	     cam_property++) {
		new_property = malloc(sizeof(V4L_PROPERTY));
		*new_property = *cam_property;
		if (cam_property->name) new_property->name = strdup(cam_property->name);
		cam_property_list = g_list_append(cam_property_list, new_property);
	}
}


V4L_PROPERTY *find_property(int cid) {
	GList *cam_property_entry;
	V4L_PROPERTY *property;

	for (cam_property_entry = cam_property_list;
	     cam_property_entry;
	     cam_property_entry = cam_property_entry->next) {
		property = (V4L_PROPERTY *)cam_property_entry->data;
		if (property->v4l2_CID == cid) {
			return(property);
		}
	}
	return(NULL);
}

V4L_PROPERTY *find_property_by_name(char *name) {
	GList *cam_property_entry;
	V4L_PROPERTY *property;

	for (cam_property_entry = cam_property_list;
	     cam_property_entry;
	     cam_property_entry = cam_property_entry->next) {
		property = (V4L_PROPERTY *)cam_property_entry->data;
		if (property->name && !strcmp(name, property->name)) {
			return(property);
		}
	}
	return(NULL);
}

int grab_init(GtkWidget *parent) {
	int rv = v4l_call[v4l_idx].grab_init(parent);

	if (rv) {
		grab_size = width * height * bits_per_pixel;
		init_pnm_buf(parent);
		exit_on_error = 0;
		cam_is_open   = 1;
	}
	return(rv);
}

void cleanup_v4l(GtkWidget *parent) {
	cam_is_open=0;
	if (v_device) {
		free (v_device);
		v_device=NULL;
	}
	if (pnm_buf) {
		free(pnm_buf);
		pnm_buf=NULL;
	}
	v4l_call[v4l_idx].cleanup(parent);
}

void get_cam_settings(GtkWidget *parent) {
	v4l_call[v4l_idx].get_cam_settings(parent);
}

void set_picture(GtkWidget *parent) {
	v4l_call[v4l_idx].set_picture(parent);
}

unsigned char* grab_one(int *x, int *y, GtkWidget *parent) {
	return(v4l_call[v4l_idx].grab_one(x, y, parent));
}

void init_pnm_buf(GtkWidget *parent)
{
	int hdr_size;

	if (pnm_buf) {
		return;
	}

	/* Allocate space for the header + the image. */
	/* header shouldn't be more than 15 bytes. Allow for 50. */
	pnm_size=grab_size+50;
	pnm_buf=malloc(pnm_size);
	if (!pnm_buf) {
		show_error_dialog("Not enough memory available.", parent);
		return;
	}

	/* The PNM header never changes, so create it only once.
	 *
	 * Write the header into the buffer, and set rgb_buf to point
	 * just past the header.
	 */

	hdr_size=sprintf(pnm_buf, "P6\n%d %d\n255\n", width, height);
	rgb_buf = pnm_buf + hdr_size;
}

/*  Image Updating Stuff */

#if defined(G_THREADS_ENABLED) || defined(USE_PTHREAD_CREATE)
int thread_started     = 0;
int thread_exit        = 0;
int upload_initialized = 0;


#ifdef G_THREADS_ENABLED
GMutex  *Captured_Images_mutex;
GCond   *Captured_Images_cond;
#else
pthread_mutex_t *Captured_Images_mutex;
pthread_cond_t  *Captured_Images_cond;
#endif

long capture_id = 0;

/* create a PNM for Jasper, convert to jpeg-2000 and send */
void *upload_image_thread(void *arg) {
	unsigned char *my_jasper=NULL;
	int my_jasper_size;
	long my_capture_id = 0;

	while (!thread_exit) {
#if defined(G_THREADS_ENABLED)
		g_mutex_lock(Captured_Images_mutex);
		if (my_capture_id == capture_id) {
			g_cond_wait(Captured_Images_cond, Captured_Images_mutex);
		}
#elif defined(USE_PTHREAD_CREATE)
		pthread_mutex_lock(Captured_Images_mutex);
		if (my_capture_id == capture_id) {
			pthread_cond_wait(Captured_Images_cond, Captured_Images_mutex);
		}
#endif

		my_capture_id = capture_id;
		if (webcam_connected && image_need_update) {
			if (app_debugger) {printf("Jasper-conversion-1\n"); fflush(stdout);}
			my_jasper = image_2_jpg(pnm_buf, pnm_size, "jpc");
			my_jasper_size = packet_size;
#if defined(G_THREADS_ENABLED)
			g_mutex_unlock(Captured_Images_mutex);
#elif defined(USE_PTHREAD_CREATE)
			pthread_mutex_unlock(Captured_Images_mutex);
#endif

			if (my_jasper) {
				if (app_debugger) {printf("Jasper-conversion-2..sending.\n"); fflush(stdout);}
				webcam_send_image(my_jasper, my_jasper_size);
				if (app_debugger) {printf("Jasper-conversion-3..sent\n"); fflush(stdout);}
				free(my_jasper); 
				my_jasper=NULL;
			}
		}
		else {
#if defined(G_THREADS_ENABLED)
			g_mutex_unlock(Captured_Images_mutex);
#elif defined(USE_PTHREAD_CREATE)
			pthread_mutex_unlock(Captured_Images_mutex);
#endif
		}
	}

#if defined(G_THREADS_ENABLED)
	g_thread_exit(0);
#elif defined(USE_PTHREAD_CREATE)
	pthread_exit(0);
#endif
	return NULL; /* satisfy compiler */
}

/* create a PNM for Jasper, convert to jpeg-2000 and send */
void *write_image_thread(void *arg) {
	unsigned char *my_jasper=NULL;
	int            my_jasper_size;
	long my_capture_id = 0;

	while (!thread_exit) {
#if defined(G_THREADS_ENABLED)
		g_mutex_lock(Captured_Images_mutex);
		if (my_capture_id == capture_id) {
			g_cond_wait(Captured_Images_cond, Captured_Images_mutex);
		}
#elif defined(USE_PTHREAD_CREATE)
		pthread_mutex_lock(Captured_Images_mutex);
		if (my_capture_id == capture_id) {
			pthread_cond_wait(Captured_Images_cond, Captured_Images_mutex);
		}
#endif

		my_capture_id = capture_id;
		if (is_recording()) {
			my_jasper = image_2_jpg(pnm_buf, pnm_size, "jpg");
			my_jasper_size = packet_size;
#if defined(G_THREADS_ENABLED)
			g_mutex_unlock(Captured_Images_mutex);
#elif defined(USE_PTHREAD_CREATE)
			pthread_mutex_unlock(Captured_Images_mutex);
#endif
			write_image_to_file(my_jasper, my_jasper_size, whoami, FrameLabel, _("Webcam Broadcaster"), main_window);
			free(my_jasper);
			my_jasper=NULL;
		}
		else {
#if defined(G_THREADS_ENABLED)
			g_mutex_unlock(Captured_Images_mutex);
#elif defined(USE_PTHREAD_CREATE)
			pthread_mutex_unlock(Captured_Images_mutex);
#endif
		}
	}

#if defined(G_THREADS_ENABLED)
	g_thread_exit(0);
#elif defined(USE_PTHREAD_CREATE)
	pthread_exit(0);
#endif
	return NULL; /* satisfy compiler */
}
#endif

void update_cam(GtkWidget *parent) {
	unsigned char *grabit;
#if defined(G_THREADS_ENABLED)
	GThread *upload_tid;
	GThread *writer_tid;
#elif defined(USE_PTHREAD_CREATE)
	pthread_t upload_tid;
	pthread_t writer_tid;
#else
	unsigned char *my_jasper=NULL;
	int            my_jasper_size;
#endif

	if (!upload_initialized) {
		init_pnm_buf(parent);

#if defined(G_THREADS_ENABLED)
		if (!thread_started) {
			Captured_Images_mutex = g_mutex_new();
			Captured_Images_cond  = g_cond_new();
			upload_tid=g_thread_create(upload_image_thread, NULL, FALSE, NULL);
			writer_tid=g_thread_create(write_image_thread,  NULL, FALSE, NULL);
			thread_started = 1;
		}
#elif defined(USE_PTHREAD_CREATE)
		if (!thread_started) {
			pthread_mutex_init(Captured_Images_mutex);
			pthread_cond_init(Captured_Images_cond);
			if (!pthread_create(&upload_tid, NULL, upload_image_thread, NULL)) {
				pthread_detach(upload_tid);
			}
			if (!pthread_create(&writer_tid, NULL, write_image_thread,  NULL)) {
				pthread_detach(writer_tid);
			}
			thread_started = 1;
		}
#endif
		upload_initialized = 1;
	}

#if defined(G_THREADS_ENABLED)
	g_mutex_lock(Captured_Images_mutex);
#elif defined(USE_PTHREAD_CREATE)
	pthread_mutex_lock(Captured_Images_mutex);
#endif
	grabit=grab_one(&width, &height, parent);
	if (grabit == NULL) {
		/* EAGAIN, or some other non-fatal error */
#if defined(G_THREADS_ENABLED)
		g_mutex_unlock(Captured_Images_mutex);
#elif defined(USE_PTHREAD_CREATE)
		pthread_mutex_unlock(Captured_Images_mutex);
#endif
		return;
	}
	if (grabit == (unsigned char *)-1) {
#if defined(G_THREADS_ENABLED)
		g_mutex_unlock(Captured_Images_mutex);
#elif defined(USE_PTHREAD_CREATE)
		pthread_mutex_unlock(Captured_Images_mutex);
#endif
		show_error_dialog("There was an error reading the webcam image.", parent);
		return;
	} 

	current_pixbuf=gdk_pixbuf_new_from_data(grabit,GDK_COLORSPACE_RGB,FALSE,8,width,height,width*24/8,NULL,NULL);
	capture_id++;

#if defined(G_THREADS_ENABLED)
	g_cond_broadcast(Captured_Images_cond);
	g_mutex_unlock(Captured_Images_mutex);
#elif defined(USE_PTHREAD_CREATE)
	pthread_cond_broadcast(Captured_Images_cond);
	pthread_mutex_unlock(Captured_Images_mutex);
#endif

	gtk_image_set_from_pixbuf(GTK_IMAGE(current_image), current_pixbuf);
	gtk_widget_show_all(current_image);
	gdk_pixbuf_unref(current_pixbuf);
	current_pixbuf=NULL;


#if defined(G_THREADS_ENABLED) ||  defined(USE_PTHREAD_CREATE)
#else
	while( gtk_events_pending()) {
		gtk_main_iteration();
	}

	if (is_recording()) {
		my_jasper = image_2_jpg(pnm_buf, pnm_size, "jpg");
		my_jasper_size = packet_size;
		write_image_to_file(my_jasper, my_jasper_size, whoami, FrameLabel, _("Webcam Broadcaster"), main_window);
		free(my_jasper);
		my_jasper=NULL;
	}

	if (webcam_connected && image_need_update) {
		init_pnm_buf(parent);
		if (rgb_buf != grabit) memcpy(rgb_buf, grabit, grab_size);
		if (app_debugger) {printf("Jasper-conversion-1\n"); fflush(stdout);}
		my_jasper= image_2_jpg(pnm_buf, pnm_size);
		my_jasper_size = packet_size;

		while( gtk_events_pending()) {
			gtk_main_iteration();
		}

		if (my_jasper) {
			if (app_debugger) {printf("Jasper-conversion-2..sending.\n"); fflush(stdout);}
			webcam_send_image(my_jasper, my_jasper_size);
			if (app_debugger) {printf("Jasper-conversion-3..sent\n"); fflush(stdout);}
			free(my_jasper);
			my_jasper=NULL;
		}
	}
#endif
}

/* signal handler */
void _sighandler (int sig) {
	GtkWidget *parent = main_window;

	switch (sig){			
		case SIGINT: /* ctrl+x */
			cleanup_v4l(parent);
			show_error_dialog("Application interrupted: Shutting down.", parent);
			break;
	}
}

void set_video_device(char *myvdev) {
	v_device=strdup(myvdev);
}

/* read webcamrc configuration file */
void read_webcamrc () {
	FILE *f;
	char line[256];

	snprintf(webcamrc_path, sizeof(webcamrc_path), 
			 "%s/.yahoorc/gyach/webcamrc", getenv("HOME"));
	
	if ((f = fopen(webcamrc_path, "r")) == NULL) {
		/* noconfig available */
		return;
	}
	while (fgets(line, sizeof(line), f)) {
		char *p, *key, *value;
		/* strip comments */
		if ((p = strchr(line, '#'))) *p = '\0';
		if ( (key = strtok(line, "=")) && (value = strtok(NULL, "\n"))) {
			if (strcmp(key,"width")==0) {
				width = atoi(value);
			}

			else if (strcmp(key,"height") == 0) {
				height = atoi(value);
			}

			else if (strcmp(key, "v4l_version") == 0) {
				if (v4l_version) free(v4l_version);
				v4l_version = strdup(value);
			}

			else {
				V4L_PROPERTY *property;
				property = find_property_by_name(key);
				if (property) property->value = atoi(value);
			}

		} else {
			fprintf(stderr, "Problems in %s\n", webcamrc_path);
		}		
	}
	fclose(f);
}

void write_webcamrc() {
	GList *cam_property_entry;
	V4L_PROPERTY *property;
	FILE *f;
	
	/* create ~/.yahoorc/gyach dir if it does not exist (it should) */
	snprintf(webcamrc_path, sizeof(webcamrc_path), "%s/.yahoorc", 
			 getenv("HOME"));
	if (access(webcamrc_path, R_OK)) {
		mkdir(webcamrc_path, 0700);
	}

	snprintf(webcamrc_path, sizeof(webcamrc_path), "%s/.yahoorc/gyach", 
			 getenv("HOME"));
	if (access(webcamrc_path, R_OK)) {
		mkdir(webcamrc_path, 0700);
	}

	snprintf(webcamrc_path, sizeof(webcamrc_path), 
			 "%s/.yahoorc/gyach/webcamrc", getenv("HOME"));

	if ((f = fopen(webcamrc_path, "w")) == NULL) {
		fprintf(stderr, "Problems when writing configuration to %s\n",
				webcamrc_path);
		return;
	}

	fprintf(f, "width=%i\n"
		   "height=%i\n"
		   "v4l_version=%s\n",
		width, height,
		gyachi_combobox_get_selected_item(v4l_combo));

	for (cam_property_entry = cam_property_list;
	     cam_property_entry;
	     cam_property_entry = cam_property_entry->next) {
		property = (V4L_PROPERTY *)cam_property_entry->data;
		if (property->value > -1) {
			fprintf(f, "%s=%i\n", property->name, property->value);
		}
	}
	fclose(f);
}


void init_cam(GtkWidget *parent) {
	char *vid_device = NULL;

	signal (SIGINT, _sighandler);

	if (!v_device) {
		show_error_dialog("No Video4Linux device was specified.", parent);
		return;
	}
	if (strlen(v_device)<1) {
		show_error_dialog("No Video4Linux device was specified.", parent);
		return;
	}
	read_webcamrc();
	/* Try to initialize the cam. If we get an error with one V4L
	 * then that means we might need to be using the other for our cam
	 * so try the other.
	 */
	if (!grab_init(parent)) {
		if (v_device) {
			vid_device=strdup(v_device);
		}
		cleanup_v4l(parent);
		if (vid_device) {
			set_video_device(vid_device);
			free(vid_device);
			vid_device = NULL;
		}

		if (v4l_idx == 1) {
			v4l_idx = 0;
			if (grab_init(parent)) {
				gyachi_combobox_select_item(v4l_combo, "V4L1");
				write_webcamrc();
			}
			else {
				/* revert back */
				v4l_idx = 1;
			}
		}
		else {
			v4l_idx = 1;
			if (grab_init(parent)) {
				gyachi_combobox_select_item(v4l_combo, "V4L2");
				write_webcamrc();
			}
			else {
				/* revert back */
				v4l_idx = 0;
			}
		}
	}

	get_cam_settings(parent);
	set_picture(parent);
}


void set_vid_properties(GtkWidget *parent) {
	set_picture(parent);
}


void select_v4l_mode(const char *new_v4l_mode, GtkWidget *parent) {
	static char *current_v4l_mode = NULL;
	char *vid_device = NULL;
	int new_idx = -1;

	/* See if any change */
	if (current_v4l_mode && new_v4l_mode && !strcmp(current_v4l_mode, new_v4l_mode)) {
		/* no change. Nothing to do */
		return;
	}

	if (new_v4l_mode) {
		if (!strcmp(new_v4l_mode, "V4L1")) new_idx = 0;
		else if (!strcmp(new_v4l_mode, "V4L2")) new_idx = 1;
	}
	if (new_idx < 0) {
		/* invalid mode/state/selection */
		return;
	}

	set_dummy_image(current_image);
	gtk_widget_show_all(current_image);

	if (current_v4l_mode) {
		/* shut down the previously selected v4l mode */
		if (v_device) {
			vid_device=strdup(v_device);
		}
		cleanup_v4l(parent);
		if (vid_device) {
			set_video_device(vid_device);
			free(vid_device);
			vid_device = NULL;
		}

		free(current_v4l_mode);
		current_v4l_mode=NULL;

		write_webcamrc();    /* update rc file with new setting */
	}

	current_v4l_mode = strdup(new_v4l_mode);
	v4l_idx = new_idx;
	init_cam(parent);
}


/* V4L2 */
int grab_init_v4l2(GtkWidget *parent) {
	struct v4l2_buffer buf;
	struct v4l2_requestbuffers req;
	enum v4l2_buf_type type;
	unsigned int min;
	unsigned int i;
	
        if ((grab_fd = v4l2_open(v_device, O_RDWR | O_NONBLOCK, 0)) == -1 ) {
		show_error_dialog("Could not open Video4Linux device.\nThe device may already be in use.", parent);
		return 0; 
	}
	
	if (v4l2_ioctl(grab_fd,VIDIOC_QUERYCAP,&grab_cap_v4l2) == -1) {
		show_error_dialog("An error occurred at 'ioctl VIDIOC_QUERYCAP'.\nNot a V4L2 device.", parent);
		return 0; 
	}
	
	if (!(grab_cap_v4l2.capabilities & V4L2_CAP_VIDEO_CAPTURE)) {
        	show_error_dialog("Fatal: grab device does not handle capture\n", parent);
		return 0;
	}
	
	if (!(grab_cap_v4l2.capabilities & V4L2_CAP_STREAMING)) {
        	show_error_dialog("Fatal: grab device does not support streaming i/o\n", parent);
		return 0;
    	}
	
	strncpy(webcam_description, grab_cap_v4l2.card, 28);

        /* Select video input, video standard and tune here. */
	
	memset (&grab_fmt, 0, sizeof(grab_fmt));
        grab_fmt.type                = V4L2_BUF_TYPE_VIDEO_CAPTURE;
        grab_fmt.fmt.pix.width       = width;
        grab_fmt.fmt.pix.height      = height;
        grab_fmt.fmt.pix.pixelformat = V4L2_PIX_FMT_BGR24;    /* This matches the v4l1 VIDEO_PALETTE_RGB24 format */
        grab_fmt.fmt.pix.field       = V4L2_FIELD_INTERLACED;

        if (v4l2_ioctl (grab_fd, VIDIOC_S_FMT, &grab_fmt) == -1) {
		show_error_dialog("Fatal: video format not supported by grab device\n", parent);
		return 0;
	}

        /* Note VIDIOC_S_FMT may change width and height. */

//	memset (&grab_fmt, 0, sizeof(grab_fmt));
//      grab_fmt.type                = V4L2_BUF_TYPE_VIDEO_CAPTURE;
//      if (v4l2_ioctl (grab_fd, VIDIOC_G_FMT, &grab_fmt) == -1) {
//		show_error_dialog("Could not get format information of grab device\n", parent);
//		return 0;
//	}
//
//	long int a = grab_fmt.fmt.pix.pixelformat;
//	char atxt[4] = "AAAA";
//	atxt[3]=(a>>24)&0xFF;
//	atxt[2]=(a>>16)&0xFF;
//	atxt[1]=(a>>8)&0xFF;
//	atxt[0]=a&0xFF;
#if 0
//	fprintf(stderr,"pixelformat: %s\n",atxt);
//	fprintf(stderr,"x: %i\n",grab_fmt.fmt.pix.width);
//	fprintf(stderr,"y: %i\n",grab_fmt.fmt.pix.height);
//	fprintf(stderr,"field: %i\n",grab_fmt.fmt.pix.field);
#endif

	/* Buggy driver paranoia. */
	min = grab_fmt.fmt.pix.width * 2;
	if (grab_fmt.fmt.pix.bytesperline < min) grab_fmt.fmt.pix.bytesperline = min;
	min = grab_fmt.fmt.pix.bytesperline * grab_fmt.fmt.pix.height;
	if (grab_fmt.fmt.pix.sizeimage < min) grab_fmt.fmt.pix.sizeimage = min;

	memset(&req, 0, sizeof(req));
        req.count  = 2;	// Doublebuffering
        req.type   = V4L2_BUF_TYPE_VIDEO_CAPTURE;
        req.memory = V4L2_MEMORY_MMAP;

	if (v4l2_ioctl (grab_fd, VIDIOC_REQBUFS, &req) == -1) {
		show_error_dialog("Fatal: memory mapping not supported by grab device\n", parent);
		return 0;
	}

        if (req.count < 2) {
		show_error_dialog("Insufficient buffer memory\n", parent);
		return 0;
        }

        buffers = calloc (req.count, sizeof (*buffers));

        if (!buffers) {
		show_error_dialog("GRAB: Out of memory\n", parent);
		return 0;
        }

        for (n_buffers = 0; n_buffers < req.count; ++n_buffers) {
		memset(&buf, 0, sizeof(buf));
                buf.type        = V4L2_BUF_TYPE_VIDEO_CAPTURE;
                buf.memory      = V4L2_MEMORY_MMAP;
                buf.index       = n_buffers;

		if (v4l2_ioctl (grab_fd, VIDIOC_QUERYBUF, &buf) == -1) {
			show_error_dialog("VIDIOC_QUERYBUF\n", parent);
			return 0;
		}
				
                buffers[n_buffers].length = buf.length;
                buffers[n_buffers].start =
                        v4l2_mmap (NULL /* start anywhere */,
                              buf.length,
                              PROT_READ | PROT_WRITE /* required */,
                              MAP_SHARED /* recommended */,
                              grab_fd, buf.m.offset);

                if (buffers[n_buffers].start == MAP_FAILED) {
			show_error_dialog("Error mmap\n", parent);
			return 0;
		}
        }
        
	/* Now turn on video streaming */
	for (i = 0; i < n_buffers; ++i) {
		memset(&buf, 0, sizeof(buf));
		buf.type        = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		buf.memory      = V4L2_MEMORY_MMAP;
		buf.index       = i;

		if (v4l2_ioctl (grab_fd, VIDIOC_QBUF, &buf) == -1) {
			show_error_dialog("VIDIOC_QBUF\n", parent);
			return(0);
		}
				
	}
		
	type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	if (v4l2_ioctl (grab_fd, VIDIOC_STREAMON, &type) == -1) {
		show_error_dialog("VIDIOC_STREAMON\n", parent);
		return(0);
	}

	return(1);
}

void cleanup_v4l2(GtkWidget *parent) {
	enum v4l2_buf_type type;
	unsigned int i;

	if (buffers) {
		for (i = 0; i < n_buffers; ++i) v4l2_munmap (buffers[i].start, buffers[i].length);
		free (buffers);
		buffers = NULL;
		n_buffers=0;
	}

	if (grab_fd > -1) {
		type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		if (v4l2_ioctl (grab_fd, VIDIOC_STREAMOFF, &type)) {
			show_error_dialog("VIDIOC_STREAMOFF\n", parent);
		}
		v4l2_close(grab_fd);
		grab_fd=-1;
	}
}

int get_control_v4l2(unsigned int id) {
	struct v4l2_queryctrl grab_ctrl;

	grab_ctrl.id = id;
	if (v4l2_ioctl(grab_fd,VIDIOC_QUERYCTRL,&grab_ctrl) != -1) {
#if 0
		fprintf(stderr, "ctrl %i:%s min:%i max:%i def:%i\n",
			id, grab_ctrl.name, grab_ctrl.minimum, grab_ctrl.maximum,
			grab_ctrl.default_value);
#endif
		return(0);
	}
	return(1);
}

void get_v4l2control(unsigned int id, V4L_PROPERTY *property) {
	struct v4l2_queryctrl grab_ctrl;
	struct v4l2_control   control;

	grab_ctrl.id = id;
	if (v4l2_ioctl(grab_fd,VIDIOC_QUERYCTRL,&grab_ctrl) != -1) {
#if 0
		fprintf(stderr, "ctrl %i:%s min:%i max:%i def:%i\n",
			id, grab_ctrl.name, grab_ctrl.minimum, grab_ctrl.maximum,
			grab_ctrl.default_value);
#endif
		property->min           = grab_ctrl.minimum;
		property->max           = grab_ctrl.maximum;
		property->default_value = grab_ctrl.default_value;
		property->step          = grab_ctrl.step;
		property->v4l_version   = 2;
		if (property->name && strcmp(property->name, grab_ctrl.name)) {
			free(property->name);
			property->name=NULL;
		}
		if (!property->name) {
			property->name=strdup(grab_ctrl.name);
		}

		control.id = id;
		if (v4l2_ioctl(grab_fd, VIDIOC_G_CTRL, &control) != -1) {
			property->value = control.value;
		}
		else {
			property->value = grab_ctrl.default_value;
		}
	}
	else {
		property->value = -1; /* not supported for this cam */
	}
}

void set_v4l2control(unsigned int id, int value) {
	struct v4l2_control propcontrol;
	
	if (!get_control_v4l2(id)) {
		propcontrol.id    = id;
		propcontrol.value = value;
		v4l2_ioctl(grab_fd, VIDIOC_S_CTRL, &propcontrol);
	}
}

void get_cam_settings_v4l2(GtkWidget *parent) {
	GList *cam_property_entry;
	V4L_PROPERTY *property;
	struct v4l2_queryctrl grab_ctrl;
	struct v4l2_control   control;
	int number_found = 0;
	int control_id;

	initialize_cam_property_list();

	for (cam_property_entry = cam_property_list;
	     cam_property_entry;
	     cam_property_entry = cam_property_entry->next) {
		property = (V4L_PROPERTY *)cam_property_entry->data;
		property->value       = -1;
		property->v4l_version = 2;
	}

#if 0
	/* This *should* work, but a lot of drivers don't yet support the NEXT_CTRL flag, yet */
	int not_done = 1;
	control.id = V4L2_CID_BASE-1;
	while (not_done) {
		control.id += 1;
		control.id |= V4L2_CTRL_FLAG_NEXT_CTRL;
		if (v4l2_ioctl(grab_fd, VIDIOC_G_CTRL, &control) != -1) {
			grab_ctrl.id = control.id;
			if (v4l2_ioctl(grab_fd,VIDIOC_QUERYCTRL,&grab_ctrl) != -1) {
#if 0
				fprintf(stderr, "ctrl %i:%s min:%i max:%i def:%i, curr: %i\n",
					control.id, grab_ctrl.name,
					grab_ctrl.minimum, grab_ctrl.maximum,
					grab_ctrl.default_value, control.value);
#endif

				property = find_property(control.id);
				if (property == 0) {
					property = malloc(sizeof(V4L_PROPERTY));
					memset(property, 0, sizeof(V4L_PROPERTY));
					cam_property_list = g_list_append(cam_property_list, property);
					property->v4l2_CID = control.id;
				}

				property->min           = grab_ctrl.minimum;
				property->max           = grab_ctrl.maximum;
				property->default_value = grab_ctrl.default_value;
				property->step          = grab_ctrl.step;
				property->v4l_version   = 2;
				property->value         = grab_ctrl.default_value;

				if (property->name && strcmp(property->name, grab_ctrl.name)) {
					free(property->name);
					property->name=NULL;
				}
				if (property->name == 0) {
					property->name = strdup(grab_ctrl.name);
				}
				number_found++;
			}
		}
		else {
			not_done = FALSE;
		}
	}

	if (number_found == 0) {
		for (cam_property_entry = cam_property_list;
		     cam_property_entry;
		     cam_property_entry = cam_property_entry->next) {
			property = (V4L_PROPERTY *)cam_property_entry->data;
			if (property->v4l2_CID > 0) {
				get_v4l2control(property->v4l2_CID, property);
			}
		}
	}

#else
	for (control_id=V4L2_CID_BASE; control_id < V4L2_CID_LASTP1; control_id++) {
		control.id = control_id;
		if (v4l2_ioctl(grab_fd, VIDIOC_G_CTRL, &control) != -1) {
			grab_ctrl.id = control.id;
			if (v4l2_ioctl(grab_fd,VIDIOC_QUERYCTRL,&grab_ctrl) != -1) {
#if 0
				fprintf(stderr, "ctrl %i:%s min:%i max:%i def:%i, curr: %i\n",
					control.id, grab_ctrl.name,
					grab_ctrl.minimum, grab_ctrl.maximum,
					grab_ctrl.default_value, control.value);
#endif

				property = find_property(control.id);
				if (property == 0) {
					property = malloc(sizeof(V4L_PROPERTY));
					memset(property, 0, sizeof(V4L_PROPERTY));
					cam_property_list = g_list_append(cam_property_list, property);
					property->v4l2_CID = control.id;
				}

				property->min           = grab_ctrl.minimum;
				property->max           = grab_ctrl.maximum;
				property->default_value = grab_ctrl.default_value;
				property->step          = grab_ctrl.step;
				property->v4l_version   = 2;
				property->value         = control.value;
				if ((property->min == 0) && (property->max == 1)) {
					property->toggle = 1;
				}

				if (property->name && strcmp(property->name, grab_ctrl.name)) {
					free(property->name);
					property->name=NULL;
				}
				if (property->name == 0) {
					property->name = strdup(grab_ctrl.name);
				}
				number_found++;
			}
		}
	}
#endif
}

void set_picture_v4l2(GtkWidget *parent) {
	GList *cam_property_entry;
	V4L_PROPERTY *property;

	for (cam_property_entry = cam_property_list;
	     cam_property_entry;
	     cam_property_entry = cam_property_entry->next) {
		property = (V4L_PROPERTY *)cam_property_entry->data;
		if ((property->value > -1 ) && (property->v4l2_CID > 0)) {
			set_v4l2control(property->v4l2_CID, property->value);
		}
	}
}

unsigned char *grab_one_v4l2(int *p_width, int *p_height, GtkWidget *parent) {
	struct v4l2_buffer buf;
	struct timeval tv;
	fd_set set;
	int ret;

	FD_ZERO( &set );
	FD_SET( grab_fd, &set );

	tv.tv_sec  = 0;
	tv.tv_usec = 0; /* poll, don't block. Check to see if buffer is ready */
	ret = select( grab_fd+1, &set, NULL, NULL, &tv );
	if (!FD_ISSET( grab_fd, &set )) {
		/* webcam not ready to be read, yet */
		return(NULL);
	}

	memset(&buf, 0, sizeof(buf));
	buf.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	buf.memory = V4L2_MEMORY_MMAP;

	if (v4l2_ioctl (grab_fd, VIDIOC_DQBUF, &buf) == -1) {
    		switch (errno) {
        	case EAGAIN:
			/* webcam not ready for a read, yet */
			return(NULL);
		default:
			show_error_dialog("grab_one: VIDIOC_DQBUF\n", parent);
			return((char *)-1);
    		}
	}

	assert (buf.index < n_buffers);
			
	grab_data=buffers[buf.index].start;

	if (v4l2_ioctl (grab_fd, VIDIOC_QBUF, &buf) == -1) {
		show_error_dialog("VIDIOC_QBUF\n", parent);
		return((char *)-1);
	}

	switch(grab_fmt.fmt.pix.pixelformat) {
	    case V4L2_PIX_FMT_SBGGR8:	bayer2rgb24 (rgb_buf, grab_data, width, height); break;
	    case V4L2_PIX_FMT_RGB332:   break;
	    case V4L2_PIX_FMT_RGB555: 	rgb555_to_rgb24(width, height, rgb_buf, grab_data); break;
	    case V4L2_PIX_FMT_RGB565:  	rgb565_to_rgb24(width, height, rgb_buf, grab_data); break;
	    case V4L2_PIX_FMT_RGB555X:  break;
	    case V4L2_PIX_FMT_RGB565X:  break;
	    case V4L2_PIX_FMT_BGR24:  
	    case V4L2_PIX_FMT_RGB24:  	rgb24_to_rgb24(width, height, rgb_buf, grab_data); break;
	    case V4L2_PIX_FMT_BGR32:  
	    case V4L2_PIX_FMT_RGB32:   	rgb32_to_rgb24(width, height, rgb_buf, grab_data); break;
	    case V4L2_PIX_FMT_GREY:	grey_to_rgb24(width, height, rgb_buf, grab_data); break;
	    case V4L2_PIX_FMT_YVU410:   break;
	    case V4L2_PIX_FMT_YVU420:   break;
	    case V4L2_PIX_FMT_YUYV:     break;
	    case V4L2_PIX_FMT_UYVY:     break;
	    case V4L2_PIX_FMT_YUV422P:  yuv422p_to_rgb24(width, height, rgb_buf, grab_data); break;
	    case V4L2_PIX_FMT_YUV411P:  break;
	    case V4L2_PIX_FMT_Y41P:     break;
	    case V4L2_PIX_FMT_NV12:     break;
	    case V4L2_PIX_FMT_NV21:     break;
	    case V4L2_PIX_FMT_YUV410:   break;
	    case V4L2_PIX_FMT_YUV420:  	yuv420p_to_rgb24(width, height, rgb_buf, grab_data); break;
	    case V4L2_PIX_FMT_YYUV:     break;
	    case V4L2_PIX_FMT_HI240:    break;
	    case V4L2_PIX_FMT_MJPEG:    break;
	    case V4L2_PIX_FMT_JPEG:     break;
	    case V4L2_PIX_FMT_DV:       break;
	    case V4L2_PIX_FMT_MPEG:     break;
	    case V4L2_PIX_FMT_WNVA:     break;
	    case V4L2_PIX_FMT_SN9C10X:  break;
	    default:                    break;
	}

	return rgb_buf;
}
/* end V4L2 */


// Here begin functions for the Video4Linux Api 1
int grab_init_v4l1(GtkWidget *parent) {
	if ((grab_fd = v4l1_open(v_device, O_RDWR)) == -1 ) {
		show_error_dialog("Could not open Video4Linux device.\nThe device may already be in use.", parent);
		return 0; 
	}

	if (v4l1_ioctl(grab_fd,VIDIOCGCAP,&grab_cap_v4l1) == -1) {
		show_error_dialog("An error occurred at 'ioctl VIDIOCGCAP'.\nWrong device.", parent);
		return 0; 
	}

	if (!grab_cap_v4l1.type & VID_TYPE_CAPTURE) {
		show_error_dialog("Device does not support capturing.", parent);
	}

	if ((grab_cap_v4l1.minwidth && (grab_cap_v4l1.minwidth > width)) ||
	    (grab_cap_v4l1.minheight && (grab_cap_v4l1.minheight > height))) {
		width = grab_cap_v4l1.minwidth;
		height = grab_cap_v4l1.minheight;
	}
		
	if ((grab_cap_v4l1.maxwidth && (grab_cap_v4l1.maxwidth < width)) ||
	    (grab_cap_v4l1.maxheight && (grab_cap_v4l1.maxheight < height))) {
		width = grab_cap_v4l1.maxwidth;
		height = grab_cap_v4l1.maxheight;
	}
		
	strncpy(webcam_description, grab_cap_v4l1.name, 28);

	memset (&grab_pic, 0, sizeof(struct video_picture));
	if (v4l1_ioctl (grab_fd, VIDIOCGPICT, &grab_pic) == -1) {
		show_error_dialog("Error getting picture information.", parent);
		return 0; 
	}

	get_cam_settings(parent);
	set_picture(parent);

	// Find out how many buffers are available
	if (v4l1_ioctl(grab_fd, VIDIOCGMBUF, &grab_mbuf) < 0)	{
		show_error_dialog("Error while querying mmap-buffers.", parent);
		return 0;
	}
	
	/* mmap all available buffers. */
	grab_data = v4l1_mmap(0, grab_mbuf.size, PROT_READ | PROT_WRITE, MAP_SHARED, grab_fd, 0);
	if (grab_data == MAP_FAILED) {
		show_error_dialog("Error mmap'ing all available buffers.", parent);
		return 0;
	}
	
	/* Setup capture options. */
	grab_buf.format = wpalette[grab_palette].palette;
	grab_buf.width  = width;
	grab_buf.height = height;
	grab_buf.frame  = 0;
	
	/* Request the maximum number of frames the device can handle. */
	if (v4l1_ioctl(grab_fd, VIDIOCMCAPTURE, &grab_buf) < 0) {
		show_error_dialog("Error while capturing initial image.", parent);
		return 0;
	}

	return(1);
}


void cleanup_v4l1(GtkWidget *parent) {
	if (grab_fd > -1) {
		v4l1_munmap(grab_data, grab_mbuf.size);
		v4l1_close (grab_fd);
		grab_fd=-1;
	}
}


void get_cam_settings_v4l1(GtkWidget *parent) {
	V4L_PROPERTY *property;
	GList *cam_property_entry;

	initialize_cam_property_list();
	memset (&grab_pic, 0, sizeof(struct video_picture));
	if (v4l1_ioctl (grab_fd, VIDIOCGPICT, &grab_pic) == -1) {
		show_error_dialog("Error getting picture information.", parent);
		return;
	}

	for (cam_property_entry = cam_property_list;
	     cam_property_entry;
	     cam_property_entry = cam_property_entry->next) {
		property = (V4L_PROPERTY *)cam_property_entry->data;

		property->min         = 0;
		property->max         = 65535;
		property->value       = -1;
		property->v4l_version = 1;

		switch (property->v4l2_CID){
		case V4L2_CID_HUE:
			property->value = grab_pic.hue;
			break;

		case V4L2_CID_CONTRAST:
			property->value = grab_pic.contrast;
			break;

		case V4L2_CID_BRIGHTNESS:
			property->value = grab_pic.brightness;
			break;

		case V4L2_CID_SATURATION:
			property->value = grab_pic.colour;
			break;

		case V4L2_CID_GAMMA:
			property->value = grab_pic.whiteness;
			break;

		default:
			break;
		}
	}
}

void set_picture_v4l1(GtkWidget *parent) {
	V4L_PROPERTY *property;

	memset(&grab_pic, 0, sizeof(grab_pic));
	property=find_property(V4L2_CID_HUE);
	if (property && property->value > -1) grab_pic.hue        = property->value;
	property=find_property(V4L2_CID_CONTRAST);
	if (property && property->value > -1) grab_pic.contrast   = property->value;
	property=find_property(V4L2_CID_BRIGHTNESS);
	if (property && property->value > -1) grab_pic.brightness = property->value;
	property=find_property(V4L2_CID_SATURATION);
	if (property && property->value > -1) grab_pic.colour     = property->value;
	property=find_property(V4L2_CID_WHITENESS);
	if (property && property->value  > -1) grab_pic.whiteness = property->value;

#if 0
	fprintf(stderr, "l4l1: hue: %08x/%d, contrast: %08x/%d, brightness: %08x/%d, colour: %08x/%d, whiteness: %08x/%d\n",
		grab_pic.hue,        grab_pic.hue,
		grab_pic.contrast,   grab_pic.contrast,
		grab_pic.brightness, grab_pic.brightness,
		grab_pic.colour,     grab_pic.colour,
		grab_pic.whiteness,  grab_pic.whiteness);
#endif

	// special case for MJPEG-devices
	if (grab_cap_v4l1.type & VID_TYPE_MJPEG_ENCODER) grab_palette = PAL_JPEG;
	
	// Try each palette 
	while(wpalette[grab_palette].name != NULL)
	{
		grab_pic.palette = wpalette[grab_palette].palette;
		grab_pic.depth   = wpalette[grab_palette].depth;

		if (!v4l1_ioctl(grab_fd, VIDIOCSPICT, &grab_pic))
		{
			return;
		}
		
		grab_palette++;
	}

	show_error_dialog("An error occurred at 'ioctl VIDIOCSPICT'.\nCould not set camera properties.", parent);
}

unsigned char* grab_one_v4l1(int *p_width, int *p_height, GtkWidget *parent) {

	set_picture_v4l1(parent);
	
	/* Wait for the frame to be captured. */
	if (v4l1_ioctl(grab_fd, VIDIOCSYNC, &grab_buf) < 0) {
		show_error_dialog("An error occurred at 'ioctl VIDIOCSYNC'.", parent);
		return((char *)-1);
	}

	if (v4l1_ioctl(grab_fd, VIDIOCMCAPTURE, &grab_buf) < 0) {
		show_error_dialog("Error while capturing an image.", parent);
		return((char *)-1);
	}

	switch(grab_palette) {
//notready	case PAL_JPEG:			jpeg_to_rgb24(width,    height, rgb_buf, grab_data, framelen); break;
		case PAL_RGB32:			rgb32_to_rgb24(width,   height, rgb_buf, grab_data);	break;
		case PAL_RGB24:			rgb24_to_rgb24(width,   height, rgb_buf, grab_data); break;
		case PAL_YUV420P:		yuv420p_to_rgb24(width, height, rgb_buf, grab_data); break;
		case PAL_YUV422P:		yuv422p_to_rgb24(width, height, rgb_buf, grab_data); break;
//old		case PAL_YUV420P:		yuv420p_to_rgb(grab_data, rgb_buf, grab_pic.depth, width, height); break;
		case PAL_RGB565:		rgb565_to_rgb24(width,  height, rgb_buf, grab_data); break;
		case PAL_RGB555:		rgb555_to_rgb24(width,  height, rgb_buf, grab_data); break;
		case PAL_GREY:			grey_to_rgb24(width,    height, rgb_buf, grab_data); break;
		default:			return NULL;
	}

	return rgb_buf;
}

/* end V4L1 */

