/*****************************************************************************
 * theme_support.c
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 *
 * Copyright (C) 2007, Greg Hosler
 * ghosler ['at'] users.sourceforge.net
 * 
 * Released under the terms of the GPL.
 * *NO WARRANTY*
 *
 *****************************************************************************/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <dirent.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include <gtk/gtk.h>

#define ICON_FACTORY
#include "theme_support.h"

/*
 * GTK_ICON_SIZE stock sizes:
 * GTK_ICON_SIZE_MENU			16x16
 * GTK_ICON_SIZE_SMALL_TOOLBAR		18x18
 * GTK_ICON_SIZE_BUTTON			20x20
 * GTK_ICON_SIZE_LARGE_TOOLBAR		24x24
 * GTK_ICON_SIZE_DND			32x32
 * GTK_ICON_SIZE_DIALOG			48x48
 *
 * We need one more for the main envelope (16x12)
 *
 */

GtkIconSize GYACHI_ICON_SIZE_MAIL;
GtkIconSize GYACHI_ICON_SIZE_LOGO;
GtkIconSize GYACHI_ICON_SIZE_VLOGO;
GtkIconSize GYACHI_ICON_SIZE_SMS;
GtkIconSize GYACHI_ICON_SIZE_LCAST;
GtkIconSize GYACHI_ICON_SIZE_PHOTO;

/* forward declarations */
void load_gyachi_icon_directory(char *theme);

/* create an icon factory using the gyachi_classic
 * theme, and then override any icons with icons
 * from a selected theme
 */
void load_gyachi_theme(char *theme)
{
	load_gyachi_icon_directory("gyachi-classic");
	if (theme && strcmp(theme, "gyachi-classic")) {
		load_gyachi_icon_directory(theme);
	}

	GYACHI_ICON_SIZE_MAIL  = gtk_icon_size_register("gyachi-icon-email",         16,  12);
	GYACHI_ICON_SIZE_LOGO  = gtk_icon_size_register("gyachi-icon-logon",         42, 235);
	GYACHI_ICON_SIZE_VLOGO = gtk_icon_size_register("gyachi-icon-voice-logon",  186,  30);
	GYACHI_ICON_SIZE_SMS   = gtk_icon_size_register("gyachi-icon-sms",           11,  17);
	GYACHI_ICON_SIZE_LCAST = gtk_icon_size_register("gyachi-icon-launchcast",    16,  13);
	GYACHI_ICON_SIZE_PHOTO = gtk_icon_size_register("gyachi-icon-photo",        100,  88);
}

GYACHI_ICON_DEF *find_icon_def(char *filename)
{
	GYACHI_ICON_DEF *icon_def;
	char *name=strdup(filename);
	char *ext;

	ext = strchr(name, '.');
	if (ext) {
		*ext = 0;
	}

	for (icon_def = gyachi_icons;
	     icon_def->filename;
	     icon_def++) {
		if (!strcmp(name, icon_def->filename)) {
			free(name);
			return(icon_def);
		}
	}
	free(name);
	return NULL;
}

void load_gyachi_icon_directory(char *theme)
{
	GtkIconFactory *icon_factory;
	GtkIconSet     *icon_set;
	GtkIconSource  *icon_source;
	char *theme_path;
	char *icon_name;
	DIR  *theme_dir;
	struct dirent *theme_dir_ent;
	GYACHI_ICON_DEF *icon_def;
	struct stat stat_buf;

	theme_path = malloc(strlen(PACKAGE_DATA_DIR) + strlen("/themes/") + strlen(theme) + 1 + 256); 
	strcpy(theme_path, PACKAGE_DATA_DIR);
	strcat(theme_path, "/themes/");
	strcat(theme_path, theme);
	strcat(theme_path, "/");
	icon_name=theme_path + strlen(theme_path);

	theme_dir = opendir(theme_path);
	if (theme_dir == 0) {
		/* look at errno for the error */

		free(theme_path);
		return;
	}

	icon_factory = gtk_icon_factory_new();
	gtk_icon_factory_add_default(icon_factory);

	/* loop thru the files in the theme directory, 
	 * handling the files that are part of our theme.
	 */
	while ((theme_dir_ent = readdir(theme_dir))) {
		icon_def = find_icon_def(theme_dir_ent->d_name);
		if (!icon_def) {
			/* don't bother checking the file if the filename's not correct */
			continue;
		}

		strcpy(icon_name, theme_dir_ent->d_name);
		if (stat(theme_path, &stat_buf)) {
			/* file doesn't stat */
			continue;
		}

		if (!S_ISREG(stat_buf.st_mode)) {
			/* not a "regular" file... */
			continue;
		}

		icon_source = gtk_icon_source_new();
		gtk_icon_source_set_filename(icon_source, theme_path);
		gtk_icon_source_set_direction_wildcarded(icon_source, TRUE);
		gtk_icon_source_set_size_wildcarded(icon_source, TRUE);
		gtk_icon_source_set_state_wildcarded(icon_source, TRUE);

		icon_set = gtk_icon_set_new();
		gtk_icon_set_add_source(icon_set, icon_source);
		gtk_icon_source_free(icon_source);

		gtk_icon_factory_add(icon_factory, icon_def->stock_id, icon_set);
		gtk_icon_set_unref(icon_set);
	}
	closedir(theme_dir);

	free(theme_path);
	g_object_unref(G_OBJECT(icon_factory));
}

/* return a sorted list of gyachi themes */
GList *gyachi_themes_available()
{
	char *theme_path;
	GList *themes = NULL;
	DIR  *theme_dir;
	struct dirent *theme_dir_ent;

	theme_path = malloc(strlen(PACKAGE_DATA_DIR) + strlen("/themes") + 1); 
	strcpy(theme_path, PACKAGE_DATA_DIR);
	strcat(theme_path, "/themes");

	theme_dir = opendir(theme_path);
	if (theme_dir == 0) {
		/* look at errno for the error */

		free(theme_path);
		return 0;
	}

	/* loop thru the theme directories */
	while ((theme_dir_ent = readdir(theme_dir))) {
		if ((theme_dir_ent->d_type != DT_DIR)   ||
		    !strcmp(theme_dir_ent->d_name, ".") ||
		    !strcmp(theme_dir_ent->d_name, "..")) {
			continue;
		}

		themes = g_list_prepend(themes, strdup(theme_dir_ent->d_name));
	}
	closedir(theme_dir);

	themes = g_list_sort(themes,  (GCompareFunc) g_utf8_collate);
	return(themes);
}

void free_gyachi_theme_list(GList *themes)
{
	while (themes) {
		free(themes->data);
		themes = g_list_delete_link(themes, themes);
	}
}

GdkPixbuf *get_pixbuf_from_stock_id(GtkWidget *w, const char *stock_id, GtkIconSize icon_size)
{
	GdkPixbuf *pixbuf;

	pixbuf = gtk_widget_render_icon (w, stock_id, icon_size, "gvwindow");

	return(pixbuf);
}
