/*****************************************************************************
 * gyachi_notebook.c
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 *
 * Copyright (C) 2008, Greg Hosler
 * (ghosler ['at'] users.sourceforge.net)
 * 
 * Released under the terms of the GPL.
 * *NO WARRANTY*
 *
 *****************************************************************************/

#include <stdlib.h>
#include <gtk/gtk.h>
#include <glib-object.h>

#include "gyachi_notebook.h"
#include "pmnotebook.h"

GSList *gyachi_notebooks = NULL;

/* forward declarations */
static gboolean gyachi_notebook_button_press_cb(GtkWidget *w, GdkEventButton *e, gpointer d);
static gboolean gyachi_notebook_button_release_cb(GtkWidget *w, GdkEventButton *e, gpointer d);
static gboolean gyachi_notebook_motion_notify_cb(GtkWidget *w, GdkEventMotion *e, gpointer d);
static gboolean toplevel_motion_notify_cb(GtkWidget *w, GdkEventMotion *event, gpointer d);
static gboolean toplevel_button_release_cb(GtkWidget *w, GdkEventButton *event, gpointer d);
static void     grab_notify_cb (GtkWidget *w, gboolean was_grabbed, gpointer d);
static gboolean grab_broken_event_cb (GtkWidget *w, GdkEventGrabBroken *e, gpointer d);

/* Local variables */
static GdkCursor *cursor = NULL;

GYACHI_NOTEBOOK_INTERFACE *gyachi_notebook_new(GtkWidget *window)
{
        GYACHI_NOTEBOOK_INTERFACE *gy_notebook=NULL;
	GtkPositionType pos = GTK_POS_TOP;

	gy_notebook = calloc(sizeof(GYACHI_NOTEBOOK_INTERFACE), 1);

	/* Create the window. */
	gy_notebook->window = window;

	/* Create the notebook. */
	gy_notebook->notebook = gtk_notebook_new();

	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(gy_notebook->notebook), pos);
	gtk_notebook_set_scrollable(GTK_NOTEBOOK(gy_notebook->notebook), TRUE);
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(gy_notebook->notebook), TRUE);
	gtk_notebook_popup_enable(GTK_NOTEBOOK(gy_notebook->notebook));
	gtk_notebook_set_show_border(GTK_NOTEBOOK(gy_notebook->notebook), FALSE);
	g_object_set(gy_notebook->notebook,"show-border", FALSE, NULL);
        g_object_set(gy_notebook->notebook,"tab-hborder", 2, NULL);
        g_object_set(gy_notebook->notebook,"tab-vborder", 1, NULL);
        g_object_set(gy_notebook->notebook,"homogeneous", TRUE, NULL);

	/* Setup the tab drag and drop signals. */
	gtk_widget_add_events(gy_notebook->notebook,
			      GDK_BUTTON1_MOTION_MASK);
	g_signal_connect(G_OBJECT(gy_notebook->notebook), "button_press_event",
			 G_CALLBACK(gyachi_notebook_button_press_cb), gy_notebook);
	g_signal_connect(G_OBJECT(gy_notebook->notebook), "button_release_event",
			 G_CALLBACK(gyachi_notebook_button_release_cb), gy_notebook);

	gyachi_notebooks = g_slist_append(gyachi_notebooks, gy_notebook);

	return(gy_notebook);
}

/* Tab Drag aNd Drop stuff */

/* The drag/drop stuff has been HEAVILY borrowed from
 * gnome-terminal, with adjustments to the implemented
 * notebook, rather than implementing a new class.
 */

#define AFTER_ALL_TABS -1
#define NOT_IN_APP_WINDOWS -2

static gboolean find_tab_num_at_pos(GtkNotebook *notebook, gint abs_x, gint abs_y)
{
	GtkPositionType tab_pos;
	int page_num = 0;
	GtkWidget *page;

	tab_pos = gtk_notebook_get_tab_pos(GTK_NOTEBOOK(notebook));

	if (GTK_NOTEBOOK(notebook)->first_tab == NULL) {
		return AFTER_ALL_TABS;
	}

	while ((page = gtk_notebook_get_nth_page(notebook, page_num))) {
		GtkWidget *screen;
		gint max_x, max_y;
		gint x_root, y_root;
  
		screen = gtk_notebook_get_tab_label(notebook, page);
		g_return_val_if_fail(screen != NULL, -1);
  
		if (!GTK_WIDGET_MAPPED(GTK_WIDGET(screen))) {
			page_num++;
			continue;
		}
  
		gdk_window_get_origin(GDK_WINDOW(screen->window),
                             &x_root, &y_root);
  
		max_x = x_root + screen->allocation.x + screen->allocation.width;
		max_y = y_root + screen->allocation.y + screen->allocation.height;
  
		if (((tab_pos == GTK_POS_TOP) || (tab_pos == GTK_POS_BOTTOM))
		    &&(abs_x<=max_x)) {
			return page_num;
		}
		else if (((tab_pos == GTK_POS_LEFT) || (tab_pos == GTK_POS_RIGHT))
			 && (abs_y<=max_y)) {
			return page_num;
		}
  
		page_num++;
	}

	return AFTER_ALL_TABS;
}


static GtkNotebook *find_notebook_at_pointer(gint abs_x, gint abs_y)
{
	GdkWindow *win_at_pointer, *toplevel_win;
	gpointer toplevel = NULL;
	gint x, y;
	GSList *l_listp;
	GYACHI_NOTEBOOK_INTERFACE *gy_notebook;

	/* FIXME multi-head */
	win_at_pointer = gdk_window_at_pointer(&x, &y);
	if (win_at_pointer == NULL) {
		/* We are outside all windows containing a notebook */
		return NULL;
	}

	toplevel_win = gdk_window_get_toplevel(win_at_pointer);

	/* search our known list of conversation/pm notebooks, looking for the
	 * one that has the same gdk_window.
	 */

	for (l_listp = gyachi_notebooks; l_listp; l_listp = l_listp->next) {
		gy_notebook = (GYACHI_NOTEBOOK_INTERFACE *)l_listp->data;
		if (gy_notebook->notebook->window == toplevel_win) {
			return(GTK_NOTEBOOK(gy_notebook->notebook));
		}
	}


	return NULL;

	/* get the GtkWidget which owns the toplevel GdkWindow */
	gdk_window_get_user_data(toplevel_win, &toplevel);

	/* toplevel should be a GtkNotebook */
	if (toplevel != NULL && GTK_IS_NOTEBOOK(toplevel)) {
		return GTK_NOTEBOOK(toplevel);
	}

	return NULL;
}

static gint find_notebook_and_tab_at_pos(gint abs_x, gint abs_y,
                                         GtkNotebook **notebook,
					 gint *page_num)
{
	*notebook = find_notebook_at_pointer(abs_x, abs_y);
	if (*notebook == NULL) {
		return NOT_IN_APP_WINDOWS;
	}
	*page_num = find_tab_num_at_pos(*notebook, abs_x, abs_y);

	if (*page_num < 0) {
		return *page_num;
	}
	else {
		return 0;
	}
}

static gboolean is_in_notebook_window(GtkNotebook *notebook,
				      gint abs_x, gint abs_y)
{
	GtkNotebook *nb_at_pointer;

	nb_at_pointer = find_notebook_at_pointer(abs_x, abs_y);

	return nb_at_pointer == notebook;
}

static void drag_stop(GYACHI_NOTEBOOK_INTERFACE *gy_notebook, guint32 time, GtkWidget *widget)
{
	GtkNotebook *notebook = GTK_NOTEBOOK(gy_notebook->notebook);
	GtkWidget   *window;
	GtkWidget   *child;

	if (gy_notebook->drag_in_progress) {
		window = gy_notebook->window;
		g_return_if_fail(GTK_WIDGET_TOPLEVEL(window));

		child = gtk_bin_get_child(GTK_BIN(window));
		g_return_if_fail(child != NULL);

		/* disconnect the signals before ungrabbing! */
		if (gy_notebook->toplevel_grab_broken_handler_id != 0) {
			g_signal_handler_disconnect(window,
						    gy_notebook->toplevel_grab_broken_handler_id);
			gy_notebook->toplevel_grab_broken_handler_id = 0;
		}
		if (gy_notebook->toplevel_motion_notify_handler_id != 0) {
			g_signal_handler_disconnect(window,
						    gy_notebook->toplevel_motion_notify_handler_id);
			gy_notebook->toplevel_motion_notify_handler_id = 0;
		}
		if (gy_notebook->toplevel_button_release_handler_id != 0) {
			g_signal_handler_disconnect(window,
						    gy_notebook->toplevel_button_release_handler_id);
			gy_notebook->toplevel_button_release_handler_id = 0;
		}
		if (gy_notebook->grab_notify_handler_id != 0) {
			g_signal_handler_disconnect(notebook,
						    gy_notebook->grab_notify_handler_id);
			gy_notebook->grab_notify_handler_id = 0;
		}

		/* ungrab the pointer if it's grabbed */
		/* FIXME multihead */
		if (gdk_pointer_is_grabbed()) {
			gdk_pointer_ungrab(GDK_CURRENT_TIME);
		}

		gtk_grab_remove(widget);
	}

	if (gy_notebook->motion_notify_handler_id != 0) {
		g_signal_handler_disconnect(notebook,
					     gy_notebook->motion_notify_handler_id);
		gy_notebook->motion_notify_handler_id = 0;
	}

	gy_notebook->drag_in_progress = FALSE;
}

static void move_tab(GYACHI_NOTEBOOK_INTERFACE *gy_notebook, int dest_position)
{
	GtkNotebook *notebook = GTK_NOTEBOOK(gy_notebook->notebook);
	gint cur_page_num;

	cur_page_num = gtk_notebook_get_current_page(notebook);

	if (dest_position != cur_page_num) {
		GtkWidget *pm_window;
      
		pm_window = gtk_notebook_get_nth_page(notebook, cur_page_num);
		gtk_notebook_reorder_child(notebook, pm_window, dest_position);
      	}
}

static gboolean drag_start(GYACHI_NOTEBOOK_INTERFACE *gy_notebook, guint32 time, GtkWidget *widget)
{
	GtkNotebook *notebook = GTK_NOTEBOOK(gy_notebook->notebook);
	GtkWidget   *window;
	GtkWidget   *child;

	/* FIXME multihead */
	if (gy_notebook->drag_in_progress || gdk_pointer_is_grabbed()) return FALSE;

	gy_notebook->drag_in_progress = TRUE;

	/* get a new cursor, if necessary */
	/* FIXME multi-head */
	if (!cursor) cursor = gdk_cursor_new(GDK_FLEUR);

	window = gy_notebook->window;
	/* sanify check */
	g_return_val_if_fail(GTK_WIDGET_TOPLEVEL(window), FALSE);

	/* sanity check*/
	child = gtk_bin_get_child(GTK_BIN(window));
	g_return_val_if_fail(child != NULL, FALSE);

	/* grab the pointer */
	gtk_grab_add(widget);

	/* FIXME multi-head */
	if (gdk_pointer_grab(window->window,
			     FALSE,
			     GDK_BUTTON1_MOTION_MASK | GDK_BUTTON_RELEASE_MASK,
			     NULL, cursor, time) != GDK_GRAB_SUCCESS) {
		drag_stop(gy_notebook, time, widget);
		return FALSE;
	}

	g_return_val_if_fail(gy_notebook->toplevel_grab_broken_handler_id == 0, FALSE);
	g_return_val_if_fail(gy_notebook->toplevel_motion_notify_handler_id == 0, FALSE);
	g_return_val_if_fail(gy_notebook->toplevel_button_release_handler_id == 0, FALSE);
	g_return_val_if_fail(gy_notebook->grab_notify_handler_id == 0, FALSE);

	gy_notebook->toplevel_grab_broken_handler_id =
		g_signal_connect(window, "grab-broken-event",
				 G_CALLBACK(grab_broken_event_cb), gy_notebook);
	gy_notebook->toplevel_motion_notify_handler_id =
		g_signal_connect(window, "motion-notify-event",
				 G_CALLBACK(toplevel_motion_notify_cb), gy_notebook);
	gy_notebook->toplevel_button_release_handler_id =
		g_signal_connect(window, "button-release-event",
				 G_CALLBACK(toplevel_button_release_cb), gy_notebook);
	gy_notebook->grab_notify_handler_id =
		g_signal_connect(notebook, "grab-notify",
				 G_CALLBACK(grab_notify_cb), gy_notebook);

	return TRUE;
}



/* callbacks on the notebook */
gboolean gyachi_notebook_button_press_cb(GtkWidget *w, GdkEventButton *event, gpointer d)
{
	GtkNotebook *notebook = GTK_NOTEBOOK(w);
	GYACHI_NOTEBOOK_INTERFACE *gy_notebook = (GYACHI_NOTEBOOK_INTERFACE *)d;

	gint tab_clicked = find_tab_num_at_pos(notebook,
					       event->x_root,
					       event->y_root);

	if (gy_notebook->drag_in_progress) {
		return TRUE;
	}
	
	if (event->type == GDK_BUTTON_PRESS) {
		if ((event->button == 1) && (tab_clicked >= 0)) {
			gy_notebook->x_start = event->x_root;
			gy_notebook->y_start = event->y_root;
			gy_notebook->motion_notify_handler_id =
			  g_signal_connect(G_OBJECT(notebook),
					    "motion-notify-event",
					    G_CALLBACK(gyachi_notebook_motion_notify_cb), gy_notebook);
		}
		else if (event->button == 3) {
			if (tab_clicked == -1) {
				/* consume event, so that we don't pop up the context menu when
				 * the mouse if not over a screen label
				 */
				return TRUE;
			}
			else {
				/* switch to the page the mouse is over, but don't consume the event */
				gtk_notebook_set_current_page(notebook, tab_clicked);
			}
		}
	}

	return FALSE;
}

gboolean gyachi_notebook_button_release_cb(GtkWidget *w, GdkEventButton *event, gpointer d)
{
	GYACHI_NOTEBOOK_INTERFACE *gy_notebook = (GYACHI_NOTEBOOK_INTERFACE *)d;

	/* This must be called even if a drag isn't happening */
	drag_stop(gy_notebook, event->time, w);

	return FALSE;
}

gboolean gyachi_notebook_motion_notify_cb(GtkWidget *w, GdkEventMotion *event, gpointer d)
{
	GtkNotebook *notebook = GTK_NOTEBOOK(w);
	GYACHI_NOTEBOOK_INTERFACE *gy_notebook = (GYACHI_NOTEBOOK_INTERFACE *)d;

	if (gy_notebook->drag_in_progress == FALSE) {
		if (gtk_drag_check_threshold(GTK_WIDGET(notebook),
					     gy_notebook->x_start,
					     gy_notebook->y_start,
					     event->x_root, event->y_root)) {
		  return drag_start(gy_notebook, event->time, w);
		}

		return FALSE;
	}

	return FALSE;
}

/* callbacks on the toplevel notebook window */
static gboolean toplevel_button_release_cb(GtkWidget *w, GdkEventButton *event, gpointer d)
{
	GYACHI_NOTEBOOK_INTERFACE *gy_notebook = (GYACHI_NOTEBOOK_INTERFACE *)d;
	GtkNotebook *notebook = GTK_NOTEBOOK(gy_notebook->notebook);

	if (gy_notebook->drag_in_progress) {
		gint cur_page_num;
		GtkWidget *cur_page;

		cur_page_num = gtk_notebook_get_current_page(notebook);
		cur_page = gtk_notebook_get_nth_page(notebook, cur_page_num);

		if (!is_in_notebook_window(notebook, event->x_root, event->y_root) &&
		    gtk_notebook_get_n_pages(GTK_NOTEBOOK(notebook)) > 1) {
			/* Tab was detached */
		}
	}

	/* This must be called even if a drag isn't happening */
	drag_stop(gy_notebook, event->time, w);

	return FALSE;
}

static gboolean grab_broken_event_cb(GtkWidget *w, GdkEventGrabBroken *e, gpointer d)
{
	GYACHI_NOTEBOOK_INTERFACE *gy_notebook = (GYACHI_NOTEBOOK_INTERFACE *)d;	

	drag_stop(gy_notebook, GDK_CURRENT_TIME, w);

	return FALSE;
}

static void grab_notify_cb(GtkWidget *w, gboolean was_grabbed, gpointer d)
{
	GYACHI_NOTEBOOK_INTERFACE *gy_notebook = (GYACHI_NOTEBOOK_INTERFACE *)d;	

	drag_stop(gy_notebook, GDK_CURRENT_TIME, w);
}

static gboolean toplevel_motion_notify_cb(GtkWidget *w, GdkEventMotion *event, gpointer d)
{
	GYACHI_NOTEBOOK_INTERFACE *gy_notebook = (GYACHI_NOTEBOOK_INTERFACE *)d;
	GtkNotebook *notebook = GTK_NOTEBOOK(gy_notebook->notebook);
	GtkNotebook *dest = NULL;
	int page_num, result;

	result = find_notebook_and_tab_at_pos((gint)event->x_root,
					      (gint)event->y_root,
					      &dest, &page_num);

	if (result != NOT_IN_APP_WINDOWS) {
		if (dest == notebook) {
			g_assert(page_num >= -1);
			move_tab(gy_notebook, page_num);
		}
	}

	return FALSE;
}
