﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/logs/CloudWatchLogsRequest.h>
#include <aws/logs/CloudWatchLogs_EXPORTS.h>

#include <utility>

namespace Aws {
namespace CloudWatchLogs {
namespace Model {

/**
 */
class DescribeMetricFiltersRequest : public CloudWatchLogsRequest {
 public:
  AWS_CLOUDWATCHLOGS_API DescribeMetricFiltersRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeMetricFilters"; }

  AWS_CLOUDWATCHLOGS_API Aws::String SerializePayload() const override;

  AWS_CLOUDWATCHLOGS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the log group.</p>
   */
  inline const Aws::String& GetLogGroupName() const { return m_logGroupName; }
  inline bool LogGroupNameHasBeenSet() const { return m_logGroupNameHasBeenSet; }
  template <typename LogGroupNameT = Aws::String>
  void SetLogGroupName(LogGroupNameT&& value) {
    m_logGroupNameHasBeenSet = true;
    m_logGroupName = std::forward<LogGroupNameT>(value);
  }
  template <typename LogGroupNameT = Aws::String>
  DescribeMetricFiltersRequest& WithLogGroupName(LogGroupNameT&& value) {
    SetLogGroupName(std::forward<LogGroupNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The prefix to match. CloudWatch Logs uses the value that you set here only if
   * you also include the <code>logGroupName</code> parameter in your request.</p>
   */
  inline const Aws::String& GetFilterNamePrefix() const { return m_filterNamePrefix; }
  inline bool FilterNamePrefixHasBeenSet() const { return m_filterNamePrefixHasBeenSet; }
  template <typename FilterNamePrefixT = Aws::String>
  void SetFilterNamePrefix(FilterNamePrefixT&& value) {
    m_filterNamePrefixHasBeenSet = true;
    m_filterNamePrefix = std::forward<FilterNamePrefixT>(value);
  }
  template <typename FilterNamePrefixT = Aws::String>
  DescribeMetricFiltersRequest& WithFilterNamePrefix(FilterNamePrefixT&& value) {
    SetFilterNamePrefix(std::forward<FilterNamePrefixT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The token for the next set of items to return. (You received this token from
   * a previous call.)</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  DescribeMetricFiltersRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of items returned. If you don't specify a value, the
   * default is up to 50 items.</p>
   */
  inline int GetLimit() const { return m_limit; }
  inline bool LimitHasBeenSet() const { return m_limitHasBeenSet; }
  inline void SetLimit(int value) {
    m_limitHasBeenSet = true;
    m_limit = value;
  }
  inline DescribeMetricFiltersRequest& WithLimit(int value) {
    SetLimit(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Filters results to include only those with the specified metric name. If you
   * include this parameter in your request, you must also include the
   * <code>metricNamespace</code> parameter.</p>
   */
  inline const Aws::String& GetMetricName() const { return m_metricName; }
  inline bool MetricNameHasBeenSet() const { return m_metricNameHasBeenSet; }
  template <typename MetricNameT = Aws::String>
  void SetMetricName(MetricNameT&& value) {
    m_metricNameHasBeenSet = true;
    m_metricName = std::forward<MetricNameT>(value);
  }
  template <typename MetricNameT = Aws::String>
  DescribeMetricFiltersRequest& WithMetricName(MetricNameT&& value) {
    SetMetricName(std::forward<MetricNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Filters results to include only those in the specified namespace. If you
   * include this parameter in your request, you must also include the
   * <code>metricName</code> parameter.</p>
   */
  inline const Aws::String& GetMetricNamespace() const { return m_metricNamespace; }
  inline bool MetricNamespaceHasBeenSet() const { return m_metricNamespaceHasBeenSet; }
  template <typename MetricNamespaceT = Aws::String>
  void SetMetricNamespace(MetricNamespaceT&& value) {
    m_metricNamespaceHasBeenSet = true;
    m_metricNamespace = std::forward<MetricNamespaceT>(value);
  }
  template <typename MetricNamespaceT = Aws::String>
  DescribeMetricFiltersRequest& WithMetricNamespace(MetricNamespaceT&& value) {
    SetMetricNamespace(std::forward<MetricNamespaceT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_logGroupName;
  bool m_logGroupNameHasBeenSet = false;

  Aws::String m_filterNamePrefix;
  bool m_filterNamePrefixHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  int m_limit{0};
  bool m_limitHasBeenSet = false;

  Aws::String m_metricName;
  bool m_metricNameHasBeenSet = false;

  Aws::String m_metricNamespace;
  bool m_metricNamespaceHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudWatchLogs
}  // namespace Aws
