﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/lakeformation/LakeFormationRequest.h>
#include <aws/lakeformation/LakeFormation_EXPORTS.h>
#include <aws/lakeformation/model/DataLakePrincipal.h>
#include <aws/lakeformation/model/DataLakeResourceType.h>
#include <aws/lakeformation/model/Resource.h>

#include <utility>

namespace Aws {
namespace LakeFormation {
namespace Model {

/**
 */
class ListPermissionsRequest : public LakeFormationRequest {
 public:
  AWS_LAKEFORMATION_API ListPermissionsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListPermissions"; }

  AWS_LAKEFORMATION_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The identifier for the Data Catalog. By default, the account ID. The Data
   * Catalog is the persistent metadata store. It contains database definitions,
   * table definitions, and other control information to manage your Lake Formation
   * environment. </p>
   */
  inline const Aws::String& GetCatalogId() const { return m_catalogId; }
  inline bool CatalogIdHasBeenSet() const { return m_catalogIdHasBeenSet; }
  template <typename CatalogIdT = Aws::String>
  void SetCatalogId(CatalogIdT&& value) {
    m_catalogIdHasBeenSet = true;
    m_catalogId = std::forward<CatalogIdT>(value);
  }
  template <typename CatalogIdT = Aws::String>
  ListPermissionsRequest& WithCatalogId(CatalogIdT&& value) {
    SetCatalogId(std::forward<CatalogIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies a principal to filter the permissions returned.</p>
   */
  inline const DataLakePrincipal& GetPrincipal() const { return m_principal; }
  inline bool PrincipalHasBeenSet() const { return m_principalHasBeenSet; }
  template <typename PrincipalT = DataLakePrincipal>
  void SetPrincipal(PrincipalT&& value) {
    m_principalHasBeenSet = true;
    m_principal = std::forward<PrincipalT>(value);
  }
  template <typename PrincipalT = DataLakePrincipal>
  ListPermissionsRequest& WithPrincipal(PrincipalT&& value) {
    SetPrincipal(std::forward<PrincipalT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies a resource type to filter the permissions returned.</p>
   */
  inline DataLakeResourceType GetResourceType() const { return m_resourceType; }
  inline bool ResourceTypeHasBeenSet() const { return m_resourceTypeHasBeenSet; }
  inline void SetResourceType(DataLakeResourceType value) {
    m_resourceTypeHasBeenSet = true;
    m_resourceType = value;
  }
  inline ListPermissionsRequest& WithResourceType(DataLakeResourceType value) {
    SetResourceType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A resource where you will get a list of the principal permissions.</p>
   * <p>This operation does not support getting privileges on a table with columns.
   * Instead, call this operation on the table, and the operation returns the table
   * and the table w columns.</p>
   */
  inline const Resource& GetResource() const { return m_resource; }
  inline bool ResourceHasBeenSet() const { return m_resourceHasBeenSet; }
  template <typename ResourceT = Resource>
  void SetResource(ResourceT&& value) {
    m_resourceHasBeenSet = true;
    m_resource = std::forward<ResourceT>(value);
  }
  template <typename ResourceT = Resource>
  ListPermissionsRequest& WithResource(ResourceT&& value) {
    SetResource(std::forward<ResourceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A continuation token, if this is not the first call to retrieve this
   * list.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListPermissionsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results to return.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListPermissionsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates that related permissions should be included in the results.</p>
   */
  inline const Aws::String& GetIncludeRelated() const { return m_includeRelated; }
  inline bool IncludeRelatedHasBeenSet() const { return m_includeRelatedHasBeenSet; }
  template <typename IncludeRelatedT = Aws::String>
  void SetIncludeRelated(IncludeRelatedT&& value) {
    m_includeRelatedHasBeenSet = true;
    m_includeRelated = std::forward<IncludeRelatedT>(value);
  }
  template <typename IncludeRelatedT = Aws::String>
  ListPermissionsRequest& WithIncludeRelated(IncludeRelatedT&& value) {
    SetIncludeRelated(std::forward<IncludeRelatedT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_catalogId;
  bool m_catalogIdHasBeenSet = false;

  DataLakePrincipal m_principal;
  bool m_principalHasBeenSet = false;

  DataLakeResourceType m_resourceType{DataLakeResourceType::NOT_SET};
  bool m_resourceTypeHasBeenSet = false;

  Resource m_resource;
  bool m_resourceHasBeenSet = false;

  Aws::String m_nextToken;
  bool m_nextTokenHasBeenSet = false;

  int m_maxResults{0};
  bool m_maxResultsHasBeenSet = false;

  Aws::String m_includeRelated;
  bool m_includeRelatedHasBeenSet = false;
};

}  // namespace Model
}  // namespace LakeFormation
}  // namespace Aws
