﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/bedrock/BedrockRequest.h>
#include <aws/bedrock/Bedrock_EXPORTS.h>
#include <aws/bedrock/model/InferenceProfileModelSource.h>
#include <aws/bedrock/model/Tag.h>
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace Bedrock {
namespace Model {

/**
 */
class CreateInferenceProfileRequest : public BedrockRequest {
 public:
  AWS_BEDROCK_API CreateInferenceProfileRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateInferenceProfile"; }

  AWS_BEDROCK_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>A name for the inference profile.</p>
   */
  inline const Aws::String& GetInferenceProfileName() const { return m_inferenceProfileName; }
  inline bool InferenceProfileNameHasBeenSet() const { return m_inferenceProfileNameHasBeenSet; }
  template <typename InferenceProfileNameT = Aws::String>
  void SetInferenceProfileName(InferenceProfileNameT&& value) {
    m_inferenceProfileNameHasBeenSet = true;
    m_inferenceProfileName = std::forward<InferenceProfileNameT>(value);
  }
  template <typename InferenceProfileNameT = Aws::String>
  CreateInferenceProfileRequest& WithInferenceProfileName(InferenceProfileNameT&& value) {
    SetInferenceProfileName(std::forward<InferenceProfileNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A description for the inference profile.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateInferenceProfileRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique, case-sensitive identifier to ensure that the API request completes
   * no more than one time. If this token matches a previous request, Amazon Bedrock
   * ignores the request, but does not return an error. For more information, see <a
   * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">Ensuring
   * idempotency</a>.</p>
   */
  inline const Aws::String& GetClientRequestToken() const { return m_clientRequestToken; }
  inline bool ClientRequestTokenHasBeenSet() const { return m_clientRequestTokenHasBeenSet; }
  template <typename ClientRequestTokenT = Aws::String>
  void SetClientRequestToken(ClientRequestTokenT&& value) {
    m_clientRequestTokenHasBeenSet = true;
    m_clientRequestToken = std::forward<ClientRequestTokenT>(value);
  }
  template <typename ClientRequestTokenT = Aws::String>
  CreateInferenceProfileRequest& WithClientRequestToken(ClientRequestTokenT&& value) {
    SetClientRequestToken(std::forward<ClientRequestTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The foundation model or system-defined inference profile that the inference
   * profile will track metrics and costs for.</p>
   */
  inline const InferenceProfileModelSource& GetModelSource() const { return m_modelSource; }
  inline bool ModelSourceHasBeenSet() const { return m_modelSourceHasBeenSet; }
  template <typename ModelSourceT = InferenceProfileModelSource>
  void SetModelSource(ModelSourceT&& value) {
    m_modelSourceHasBeenSet = true;
    m_modelSource = std::forward<ModelSourceT>(value);
  }
  template <typename ModelSourceT = InferenceProfileModelSource>
  CreateInferenceProfileRequest& WithModelSource(ModelSourceT&& value) {
    SetModelSource(std::forward<ModelSourceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An array of objects, each of which contains a tag and its value. For more
   * information, see <a
   * href="https://docs.aws.amazon.com/bedrock/latest/userguide/what-is-service.html">Tagging
   * resources</a> in the <a
   * href="https://docs.aws.amazon.com/bedrock/latest/userguide/what-is-service.html">Amazon
   * Bedrock User Guide</a>.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateInferenceProfileRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateInferenceProfileRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_inferenceProfileName;
  bool m_inferenceProfileNameHasBeenSet = false;

  Aws::String m_description;
  bool m_descriptionHasBeenSet = false;

  Aws::String m_clientRequestToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientRequestTokenHasBeenSet = true;

  InferenceProfileModelSource m_modelSource;
  bool m_modelSourceHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Bedrock
}  // namespace Aws
