﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/acm-pca/ACMPCARequest.h>
#include <aws/acm-pca/ACMPCA_EXPORTS.h>
#include <aws/acm-pca/model/Tag.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace ACMPCA {
namespace Model {

/**
 */
class UntagCertificateAuthorityRequest : public ACMPCARequest {
 public:
  AWS_ACMPCA_API UntagCertificateAuthorityRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UntagCertificateAuthority"; }

  AWS_ACMPCA_API Aws::String SerializePayload() const override;

  AWS_ACMPCA_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) that was returned when you called <a
   * href="https://docs.aws.amazon.com/privateca/latest/APIReference/API_CreateCertificateAuthority.html">CreateCertificateAuthority</a>.
   * This must be of the form: </p> <p>
   * <code>arn:aws:acm-pca:<i>region</i>:<i>account</i>:certificate-authority/<i>12345678-1234-1234-1234-123456789012</i>
   * </code> </p>
   */
  inline const Aws::String& GetCertificateAuthorityArn() const { return m_certificateAuthorityArn; }
  inline bool CertificateAuthorityArnHasBeenSet() const { return m_certificateAuthorityArnHasBeenSet; }
  template <typename CertificateAuthorityArnT = Aws::String>
  void SetCertificateAuthorityArn(CertificateAuthorityArnT&& value) {
    m_certificateAuthorityArnHasBeenSet = true;
    m_certificateAuthorityArn = std::forward<CertificateAuthorityArnT>(value);
  }
  template <typename CertificateAuthorityArnT = Aws::String>
  UntagCertificateAuthorityRequest& WithCertificateAuthorityArn(CertificateAuthorityArnT&& value) {
    SetCertificateAuthorityArn(std::forward<CertificateAuthorityArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>List of tags to be removed from the CA.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  UntagCertificateAuthorityRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  UntagCertificateAuthorityRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_certificateAuthorityArn;
  bool m_certificateAuthorityArnHasBeenSet = false;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace ACMPCA
}  // namespace Aws
