import {
  AbsoluteTime,
  assertUnreachable,
  Duration,
  OperationAlternative,
  OperationFail,
  OperationOk,
  OperationResult,
  TalerError,
  TalerErrorCode,
  TranslatedString,
} from "@gnu-taler/taler-util";
import { useEffect, useState } from "preact/hooks";
import { ButtonHandler } from "../components/Button.js";
import {
  InternationalizationAPI,
  memoryMap,
  useTranslationContext,
} from "../index.browser.js";

export type NotificationMessage = ErrorNotification | InfoNotification;

export interface ErrorNotification {
  type: "error";
  title: TranslatedString;
  ack?: boolean;
  timeout?: boolean;
  description?: TranslatedString;
  debug?: any;
  when: AbsoluteTime;
}
export interface InfoNotification {
  type: "info";
  title: TranslatedString;
  ack?: boolean;
  timeout?: boolean;
  when: AbsoluteTime;
}

const storage = memoryMap<Map<string, NotificationMessage>>();
const NOTIFICATION_KEY = "notification";

export const GLOBAL_NOTIFICATION_TIMEOUT = Duration.fromSpec({
  seconds: 5,
});

function updateInStorage(n: NotificationMessage) {
  const h = hash(n);
  const mem = storage.get(NOTIFICATION_KEY) ?? new Map();
  const newState = new Map(mem);
  newState.set(h, n);
  storage.set(NOTIFICATION_KEY, newState);
}

export function notify(notif: NotificationMessage): void {
  const currentState: Map<string, NotificationMessage> =
    storage.get(NOTIFICATION_KEY) ?? new Map();
  const newState = currentState.set(hash(notif), notif);

  if (GLOBAL_NOTIFICATION_TIMEOUT.d_ms !== "forever") {
    setTimeout(() => {
      notif.timeout = true;
      updateInStorage(notif);
    }, GLOBAL_NOTIFICATION_TIMEOUT.d_ms);
  }

  storage.set(NOTIFICATION_KEY, newState);
}
export function notifyError(
  title: TranslatedString,
  description: TranslatedString | undefined,
  debug?: any,
) {
  notify({
    type: "error" as const,
    title,
    description,
    debug,
    when: AbsoluteTime.now(),
  });
}
export function notifyException(title: TranslatedString, ex: Error) {
  notify({
    type: "error" as const,
    title,
    description: ex.message as TranslatedString,
    debug: ex.stack,
    when: AbsoluteTime.now(),
  });
}
export function notifyInfo(title: TranslatedString) {
  notify({
    type: "info" as const,
    title,
    when: AbsoluteTime.now(),
  });
}

export type Notification = {
  message: NotificationMessage;
  acknowledge: () => void;
};

export function useNotifications(): Notification[] {
  const [, setLastUpdate] = useState<number>();
  const value = storage.get(NOTIFICATION_KEY) ?? new Map();

  useEffect(() => {
    return storage.onUpdate(NOTIFICATION_KEY, () => {
      setLastUpdate(Date.now());
      // const mem = storage.get(NOTIFICATION_KEY) ?? new Map();
      // setter(structuredClone(mem));
    });
  });

  return Array.from(value.values()).map((message, idx) => {
    return {
      message,
      acknowledge: () => {
        message.ack = true;
        updateInStorage(message);
      },
    };
  });
}

function hashCode(str: string): string {
  if (str.length === 0) return "0";
  let hash = 0;
  let chr;
  for (let i = 0; i < str.length; i++) {
    chr = str.charCodeAt(i);
    hash = (hash << 5) - hash + chr;
    hash |= 0; // Convert to 32bit integer
  }
  return hash.toString(16);
}

function hash(msg: NotificationMessage): string {
  let str = (msg.type + ":" + msg.title) as string;
  if (msg.type === "error") {
    if (msg.description) {
      str += ":" + msg.description;
    }
    if (msg.debug) {
      str += ":" + msg.debug;
    }
  }
  return hashCode(str);
}

function errorMap<T extends OperationFail<unknown>>(
  resp: T,
  map: (d: T["case"]) => TranslatedString,
): void {
  notify({
    type: "error",
    title: map(resp.case),
    description: (resp.detail?.hint as TranslatedString) ?? "",
    debug: resp.detail,
    when: AbsoluteTime.now(),
  });
}

export type ErrorNotificationHandler = (
  cb: (notify: typeof errorMap) => Promise<void>,
) => Promise<void>;

/**
 * @deprecated use useLocalNotificationHandler
 *
 * @returns
 */
export function useLocalNotification(): [
  Notification | undefined,
  (n: NotificationMessage) => void,
  ErrorNotificationHandler,
] {
  const { i18n } = useTranslationContext();

  const [value, setter] = useState<NotificationMessage>();
  const notif = !value
    ? undefined
    : {
        message: value,
        acknowledge: () => {
          setter(undefined);
        },
      };

  async function errorHandling(cb: (notify: typeof errorMap) => Promise<void>) {
    try {
      return await cb(errorMap);
    } catch (error: unknown) {
      if (error instanceof TalerError) {
        notify(buildUnifiedRequestErrorMessage(i18n, error));
      } else {
        notifyError(
          i18n.str`Operation failed, please report`,
          (error instanceof Error
            ? error.message
            : JSON.stringify(error)) as TranslatedString,
        );
      }
    }
  }
  return [notif, setter, errorHandling];
}

type HandlerMaker = <K extends any[], T extends OperationResult<A, B>, A, B>(
  onClick: (...args: K) => Promise<T>,
  onOperationSuccess: OnOperationSuccesReturnType<T, K>,
  onOperationFail?: OnOperationFailReturnType<T, K>,
  onOperationComplete?: () => void,
) => ButtonHandler;

/**
 * @deprecated use useLocalNotificationBetter
 * @returns
 */
export function useLocalNotificationHandler(): [
  Notification | undefined,
  HandlerMaker,
  (n: NotificationMessage) => void,
] {
  const { i18n } = useTranslationContext();
  const [value, setter] = useState<NotificationMessage>();
  const notif = !value
    ? undefined
    : {
        message: value,
        acknowledge: () => {
          setter(undefined);
        },
      };

  function makeHandler<K extends any[], T extends OperationResult<A, B>, A, B>(
    doAction: (...args: K) => Promise<T>,
    onOperationSuccess: OnOperationSuccesReturnType<T, K>,
    onOperationFail?: OnOperationFailReturnType<T, K>,
    onOperationComplete?: () => void,
  ): ButtonHandler {
    const onNotification = setter;
    return {
      onClick: async (...args: K): Promise<void> => {
        try {
          const resp = await doAction(...args);
          if (resp) {
            if (resp.type === "ok") {
              const result: OperationOk<any> = resp;
              // @ts-expect-error this is an operationOk
              const msg = onOperationSuccess(result, ...args);
              if (msg) {
                notifyInfo(msg);
              }
            }
            if (resp.type === "fail") {
              const d = "detail" in resp ? resp.detail : undefined;

              const title = !onOperationFail
                ? i18n.str`Unexpected error`
                : onOperationFail(resp as any, ...args);
              onNotification({
                title,
                type: "error",
                description:
                  d && d.hint ? (d.hint as TranslatedString) : undefined,
                debug: d,
                when: AbsoluteTime.now(),
              });
            }
          }
          if (onOperationComplete) {
            onOperationComplete();
          }
          return;
        } catch (error: unknown) {
          console.error(error);

          if (error instanceof TalerError) {
            onNotification(buildUnifiedRequestErrorMessage(i18n, error));
          } else {
            const description = (
              error instanceof Error ? error.message : String(error)
            ) as TranslatedString;

            onNotification({
              title: i18n.str`Operation failed`,
              type: "error",
              description,
              when: AbsoluteTime.now(),
            });
          }
          if (onOperationComplete) {
            onOperationComplete();
          }
          return;
        }
        // setRunning(false);
      },
    };
  }

  return [notif, makeHandler, setter];
}

// type HandlerMakerBetter = <
//   K extends any[],
//   T extends OperationResult<A, B>,
//   A,
//   B,
// >(
//   onClick: (...args: K) => Promise<T>,
//   onOperationSuccess: OnOperationSuccesReturnType<T, K>,
//   onOperationFail?: OnOperationFailReturnType<T, K>,
//   onOperationComplete?: () => void,
// ) => (...args: K) => Promise<void>;

// export function useLocalNotificationBetter(): [
//   Notification | undefined,
//   HandlerMakerBetter,
//   (n: NotificationMessage) => void,
// ] {
//   const { i18n } = useTranslationContext();
//   const [value, setter] = useState<NotificationMessage>();
//   const notif = !value
//     ? undefined
//     : {
//         message: value,
//         acknowledge: () => {
//           setter(undefined);
//         },
//       };

//   function makeHandler<K extends any[], T extends OperationResult<A, B>, A, B>(
//     doAction: (...args: K) => Promise<T>,
//     onOperationSuccess: OnOperationSuccesReturnType<T, K>,
//     onOperationFail?: OnOperationFailReturnType<T, K>,
//     onOperationComplete?: () => void,
//   ): () => Promise<void> {
//     const onNotification = setter;
//     return async (...args: K): Promise<void> => {
//       try {
//         const resp = await doAction(...args);
//         if (resp) {
//           if (resp.type === "ok") {
//             const result: OperationOk<any> = resp;
//             // @ts-expect-error this is an operationOk
//             const msg = onOperationSuccess(result, ...args);
//             if (msg) {
//               notifyInfo(msg);
//             }
//           }
//           if (resp.type === "fail") {
//             const d = "detail" in resp ? resp.detail : undefined;

//             const title = !onOperationFail
//               ? i18n.str`Unexpected error`
//               : onOperationFail(resp as any, ...args);
//             onNotification({
//               title,
//               type: "error",
//               description:
//                 d && d.hint ? (d.hint as TranslatedString) : undefined,
//               debug: d,
//               when: AbsoluteTime.now(),
//             });
//           }
//         }
//         if (onOperationComplete) {
//           onOperationComplete();
//         }
//         return;
//       } catch (error: unknown) {
//         console.error(error);

//         if (error instanceof TalerError) {
//           onNotification(buildUnifiedRequestErrorMessage(i18n, error));
//         } else {
//           const description = (
//             error instanceof Error ? error.message : String(error)
//           ) as TranslatedString;

//           onNotification({
//             title: i18n.str`Operation failed`,
//             type: "error",
//             description,
//             when: AbsoluteTime.now(),
//           });
//         }
//         if (onOperationComplete) {
//           onOperationComplete();
//         }
//         return;
//       }
//       // setRunning(false);
//     };
//   }

//   return [notif, makeHandler, setter];
// }

function buildUnifiedRequestErrorMessage(
  i18n: InternationalizationAPI,
  cause: TalerError,
): ErrorNotification {
  let result: ErrorNotification;
  switch (cause.errorDetail.code) {
    case TalerErrorCode.GENERIC_TIMEOUT: {
      result = {
        type: "error",
        title: i18n.str`Request timeout`,
        description: cause.message as TranslatedString,
        debug: cause.errorDetail,
        when: AbsoluteTime.now(),
      };
      break;
    }
    case TalerErrorCode.GENERIC_CLIENT_INTERNAL_ERROR: {
      result = {
        type: "error",
        title: i18n.str`Request cancelled`,
        description: cause.message as TranslatedString,
        debug: cause.errorDetail,
        when: AbsoluteTime.now(),
      };
      break;
    }
    case TalerErrorCode.WALLET_HTTP_REQUEST_GENERIC_TIMEOUT: {
      result = {
        type: "error",
        title: i18n.str`Request timeout`,
        description: cause.message as TranslatedString,
        debug: cause.errorDetail,
        when: AbsoluteTime.now(),
      };
      break;
    }
    case TalerErrorCode.WALLET_HTTP_REQUEST_THROTTLED: {
      result = {
        type: "error",
        title: i18n.str`Request throttled`,
        description: cause.message as TranslatedString,
        debug: cause.errorDetail,
        when: AbsoluteTime.now(),
      };
      break;
    }
    case TalerErrorCode.WALLET_RECEIVED_MALFORMED_RESPONSE: {
      result = {
        type: "error",
        title: i18n.str`Malformed response`,
        description: cause.message as TranslatedString,
        debug: cause.errorDetail,
        when: AbsoluteTime.now(),
      };
      break;
    }
    case TalerErrorCode.WALLET_NETWORK_ERROR: {
      result = {
        type: "error",
        title: i18n.str`Network error`,
        description: cause.message as TranslatedString,
        debug: cause.errorDetail,
        when: AbsoluteTime.now(),
      };
      break;
    }
    case TalerErrorCode.WALLET_UNEXPECTED_REQUEST_ERROR: {
      result = {
        type: "error",
        title: i18n.str`Unexpected request error`,
        description: cause.message as TranslatedString,
        debug: cause.errorDetail,
        when: AbsoluteTime.now(),
      };
      break;
    }
    default: {
      result = {
        type: "error",
        title: i18n.str`Unexpected error`,
        description: cause.message as TranslatedString,
        debug: cause.errorDetail,
        when: AbsoluteTime.now(),
      };
      break;
    }
  }
  return result;
}

export type FunctionThatMayFail<T extends any[]> = (
  ...args: T
) => Promise<NotificationMessage | undefined>;

type FunctionWrapperForButton = <T extends any[]>(
  fn: FunctionThatMayFail<T>,
) => (...args: T) => Promise<void>;

export function useLocalNotificationBetter(): [
  Notification | undefined,
  FunctionWrapperForButton,
] {
  const [value, setter] = useState<NotificationMessage>();
  const notif = !value
    ? undefined
    : {
        message: value,
        acknowledge: () => {
          setter(undefined);
        },
      };

  function wrapForButtons<T extends any[]>(
    fn: (...args: T) => Promise<NotificationMessage | undefined>,
  ): (...args: T) => Promise<void> {
    return async (...params: T): Promise<void> => {
      const error = await fn(...params);
      if (error) {
        setter(error);
      }
    };
  }

  return [notif, wrapForButtons];
}

/**
 * Convert an function that return an operation into a function that return
 * a notification if it fail.
 *
 * @param i18n
 * @param doAction
 * @param onOperationSuccess
 * @param onOperationFail
 * @returns
 */
export function makeSafeCall<
  K extends any[],
  T extends OperationResult<A, B>,
  A,
  B,
>(
  i18n: InternationalizationAPI,
  doAction: (...args: K) => Promise<T>,
  onOperationSuccess: OnOperationSuccesReturnType<T, K>,
  onOperationFail?: OnOperationFailReturnType<T, K>,
): FunctionThatMayFail<K> {
  return async (...args: K): Promise<NotificationMessage | undefined> => {
    try {
      const resp = await doAction(...args);
      switch (resp.type) {
        case "ok": {
          const result: OperationOk<any> = resp;
          // @ts-expect-error this is an operationOk
          const msg = onOperationSuccess(result, ...args);
          if (msg) {
            notifyInfo(msg);
          }
          return undefined;
        }
        case "fail": {
          const d = "detail" in resp ? resp.detail : undefined;

          const title = !onOperationFail
            ? i18n.str`Unexpected error`
            : onOperationFail(resp as any, ...args);
          return {
            title,
            type: "error",
            description: d && d.hint ? (d.hint as TranslatedString) : undefined,
            debug: d,
            when: AbsoluteTime.now(),
          };
        }
        default: {
          assertUnreachable(resp);
        }
      }
    } catch (error: unknown) {
      console.error(error);

      if (error instanceof TalerError) {
        return buildUnifiedRequestErrorMessage(i18n, error);
      } else {
        const description = (
          error instanceof Error ? error.message : String(error)
        ) as TranslatedString;

        return {
          title: i18n.str`Operation failed`,
          type: "error",
          description,
          when: AbsoluteTime.now(),
        };
      }
    }
  };
}

export interface SafeHandler<
  K extends any[],
  T extends OperationResult<A, B>,
  A,
  B,
> {
  (...args: K): Promise<NotificationMessage | undefined>;
  onSuccess: OnOperationSuccesReturnType<T, K>;
  onFail: OnOperationFailReturnType<T, K>;
  onUnexpectedFailure: OnOperationUnexpectedFailReturnType<K>;
}

/**
 * Convert an function that return an operation into a function that return
 * a notification if it fail.
 *
 * @returns
 */
export function safeFunctionCall<
  K extends any[],
  T extends OperationResult<A, B>,
  A,
  B,
>(doAction: (...args: K) => Promise<T>): SafeHandler<K, T, A, B> {
  const handler = (async (
    ...args: K
  ): Promise<NotificationMessage | undefined> => {
    try {
      const resp = await doAction(...args);
      switch (resp.type) {
        case "ok": {
          const result: OperationOk<any> = resp;
          const msg = handler.onSuccess(result as any, ...args);
          if (msg) {
            notifyInfo(msg);
          }
          return undefined;
        }
        case "fail": {
          const d = "detail" in resp ? resp.detail : undefined;
          const title = handler.onFail(resp as any, ...args);
          return {
            title,
            type: "error",
            description: d && d.hint ? (d.hint as TranslatedString) : undefined,
            debug: d,
            when: AbsoluteTime.now(),
          };
        }
        default: {
          assertUnreachable(resp);
        }
      }
    } catch (error: unknown) {
      // This functions should not throw, this is a problem.
      console.error(`Error: `, error);

      if (error instanceof TalerError) {
        return {
          title: handler.onUnexpectedFailure(error, ...args),
          type: "error",
          description:
            error && error.errorDetail.hint
              ? (error.errorDetail.hint as TranslatedString)
              : undefined,
          debug: error,
          when: AbsoluteTime.now(),
        };
      } else {
        const description = (
          error instanceof Error ? error.message : String(error)
        ) as TranslatedString;

        return {
          title: `Operation failed` as TranslatedString,
          type: "error",
          description,
          when: AbsoluteTime.now(),
        };
      }
    }
  }) as SafeHandler<K, T, A, B>;
  handler.onFail = () => "<unhandled failure>" as TranslatedString;
  handler.onSuccess = () => {};
  handler.onUnexpectedFailure = () =>
    "<unhandled unexpected failure>" as TranslatedString;
  return handler;
}

export type OnOperationSuccesReturnType<T, K extends any[]> = (
  result: T extends OperationOk<any> ? T : never,
  ...args: K
) => TranslatedString | void;

export type OnOperationFailReturnType<T, K extends any[]> = (
  d:
    | (T extends OperationFail<any> ? T : never)
    | (T extends OperationAlternative<any, any> ? T : never),
  ...args: K
) => TranslatedString;

export type OnOperationUnexpectedFailReturnType<K extends any[]> = (
  e: TalerError,
  ...args: K
) => TranslatedString;
