/*
 This file is part of GNU Taler
 (C) 2020-2025 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  ExchangeUpdateStatus,
  TalerErrorCode,
  j2s,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { defaultCoinConfig } from "../harness/denomStructures.js";
import {
  ExchangeService,
  GlobalTestState,
  setupDb,
} from "../harness/harness.js";
import {
  createSimpleTestkudosEnvironmentV3,
  withdrawViaBankV3,
} from "../harness/environments.js";

/**
 * Test the wallet's behavior when the exchange switches to a completely
 * new master public keyy.
 */
export async function runExchangeMasterPubChangeTest(
  t: GlobalTestState,
): Promise<void> {
  // Set up test environment

  const { walletClient, exchange, bankClient, exchangeBankAccount } =
    await createSimpleTestkudosEnvironmentV3(t);

  const wres = await withdrawViaBankV3(t, {
    walletClient,
    amount: "TESTKUDOS:10",
    bankClient,
    exchange,
  });

  await wres.withdrawalFinishedCond;

  t.logStep("withdrawal-done");

  const exchangesListOld = await walletClient.call(
    WalletApiOperation.ListExchanges,
    {},
  );

  console.log(j2s(exchangesListOld));

  await exchange.stop();

  // Instead of reconfiguring the old exchange, we just create a new exchange here
  // that runs under the same base URL as the old exchange.

  const db2 = await setupDb(t, {
    nameSuffix: "e2",
  });
  const exchange2 = ExchangeService.create(t, {
    name: "testexchange-2",
    currency: "TESTKUDOS",
    httpPort: 8081,
    database: db2.connStr,
  });

  await exchange2.addBankAccount("1", exchangeBankAccount);
  exchange2.addCoinConfigList(defaultCoinConfig.map((x) => x("TESTKUDOS")));
  await exchange2.start();

  t.logStep("exchange-restarted");

  const err = await t.assertThrowsTalerErrorAsyncLegacy(
    walletClient.call(WalletApiOperation.UpdateExchangeEntry, {
      exchangeBaseUrl: exchange.baseUrl,
      force: true,
    })
  );

  console.log("updateExchangeEntry err:", j2s(err));

  const exchangesList = await walletClient.call(
    WalletApiOperation.ListExchanges,
    {},
  );

  console.log(j2s(exchangesList));

  t.assertDeepEqual(
    exchangesList.exchanges[0].exchangeUpdateStatus,
    ExchangeUpdateStatus.UnavailableUpdate,
  );
  t.assertDeepEqual(
    exchangesList.exchanges[0].unavailableReason?.code,
    TalerErrorCode.WALLET_EXCHANGE_ENTRY_UPDATE_CONFLICT,
  );
}

runExchangeMasterPubChangeTest.suites = ["wallet", "exchange"];
