/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2020 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Merge      mergetime       Merge datasets sorted by date and time
*/

#include <cdi.h>

#include "cdo_options.h"
#include "cdo_rlimit.h"
#include "functs.h"
#include "process_int.h"
#include "cdo_vlist.h"
#include "util_files.h"
#include "printinfo.h"

struct StreamFile
{
  CdoStreamID streamID;
  int vlistID;
  int taxisID;
  int tsID;
  int64_t vdate;
  int vtime;
  int nrecs;
  VarList varList;
};

bool
getenvSkipSameTime()
{
  const auto envstr = getenv("SKIP_SAME_TIME");
  if (envstr)
    {
      const auto ival = atoi(envstr);
      if (ival == 1)
        {
          if (Options::cdoVerbose) cdoPrint("Set SKIP_SAME_TIME to %d", ival);
          return true;
        }
    }

  return false;
}

static void
openAllFiles(int nfiles, std::vector<StreamFile> &sf)
{
  for (int fileID = 0; fileID < nfiles; fileID++)
    {
      if (Options::cdoVerbose) cdoPrint("process: %s", cdoGetStreamName(fileID));

      auto &s = sf[fileID];
      s.streamID = cdoOpenRead(fileID);
      s.vlistID = cdoStreamInqVlist(s.streamID);
      s.taxisID = vlistInqTaxis(s.vlistID);
      varListInit(s.varList, s.vlistID);
    }
}

static void
readFirstTimestep(int nfiles, std::vector<StreamFile> &sf)
{
  for (int fileID = 0; fileID < nfiles; fileID++)
    {
      auto &s = sf[fileID];
      s.tsID = 0;
      s.nrecs = cdoStreamInqTimestep(s.streamID, s.tsID);
      if (s.nrecs == 0)
        {
          cdoStreamClose(s.streamID);
          s.streamID = CDO_STREAM_UNDEF;
        }
      else
        {
          s.vdate = taxisInqVdate(s.taxisID);
          s.vtime = taxisInqVtime(s.taxisID);
        }
    }
}

void *
Mergetime(void *process)
{
  int tsID2 = 0;
  int taxisID2 = CDI_UNDEFID;
  int64_t last_vdate = -1;
  int last_vtime = -1;

  cdoInitialize(process);

  operatorCheckArgc(0);

  const auto skipSameTime = getenvSkipSameTime();

  const auto lcopy = unchangedRecord();

  const auto nfiles = cdoStreamCnt() - 1;
  std::vector<StreamFile> sf(nfiles);

  cdo::set_numfiles(nfiles + 8);

  openAllFiles(nfiles, sf);

  // check that the contents is always the same
  for (int fileID = 1; fileID < nfiles; fileID++) vlistCompare(sf[0].vlistID, sf[fileID].vlistID, CMP_ALL);

  // read the first time step
  readFirstTimestep(nfiles, sf);

  auto ofilename = cdoGetStreamName(nfiles);
  if (!Options::cdoOverwriteMode && fileExists(ofilename) && !userFileOverwrite(ofilename))
    cdoAbort("Outputfile %s already exists!", ofilename);

  const auto streamID2 = cdoOpenWrite(nfiles);

  Field field;

  while (true)
    {
      bool process_timestep = true;

      int next_fileID = -1;
      int64_t vdate = 0;
      int vtime = 0;
      for (int fileID = 0; fileID < nfiles; fileID++)
        {
          if (sf[fileID].streamID != CDO_STREAM_UNDEF)
            if (next_fileID == -1 || sf[fileID].vdate < vdate || (sf[fileID].vdate == vdate && sf[fileID].vtime < vtime))
              {
                next_fileID = fileID;
                vdate = sf[fileID].vdate;
                vtime = sf[fileID].vtime;
              }
        }

      const auto fileID = next_fileID;
      if (Options::cdoVerbose) cdoPrint("nextstep = %d  vdate = %ld  vtime = %d", fileID, vdate, vtime);
      if (fileID == -1) break;

      auto &s = sf[fileID];

      if (skipSameTime && vdate == last_vdate && vtime == last_vtime)
        {
          cdoPrint("Timestep %4d in stream %d (%s %s) already exists, skipped!", s.tsID + 1,
                   s.streamID->getID(), dateToString(vdate).c_str(), timeToString(vtime).c_str());
          process_timestep = false;
        }

      if (process_timestep)
        {
          if (tsID2 == 0)
            {
              const auto vlistID1 = s.vlistID;
              const auto vlistID2 = vlistDuplicate(vlistID1);
              const auto taxisID1 = vlistInqTaxis(vlistID1);
              taxisID2 = taxisDuplicate(taxisID1);
              vlistDefTaxis(vlistID2, taxisID2);

              cdoDefVlist(streamID2, vlistID2);
            }

          last_vdate = vdate;
          last_vtime = vtime;

          taxisCopyTimestep(taxisID2, s.taxisID);
          cdoDefTimestep(streamID2, tsID2);

          for (int recID = 0; recID < s.nrecs; recID++)
            {
              int varID, levelID;
              cdoInqRecord(s.streamID, &varID, &levelID);

              if (tsID2 > 0 && s.tsID == 0 && s.varList[varID].timetype == TIME_CONSTANT) continue;

              cdoDefRecord(streamID2, varID, levelID);

              if (lcopy)
                {
                  cdoCopyRecord(streamID2, s.streamID);
                }
              else
                {
                  field.init(s.varList[varID]);
                  cdoReadRecord(s.streamID, field);
                  cdoWriteRecord(streamID2, field);
                }
            }

          tsID2++;
        }

      s.nrecs = cdoStreamInqTimestep(s.streamID, ++s.tsID);
      if (s.nrecs == 0)
        {
          cdoStreamClose(s.streamID);
          s.streamID = CDO_STREAM_UNDEF;
        }
      else
        {
          s.vdate = taxisInqVdate(s.taxisID);
          s.vtime = taxisInqVtime(s.taxisID);
        }
    }

  cdoStreamClose(streamID2);

  cdoFinish();

  return nullptr;
}
